#include <stdio.h>
#include <stdbool.h>
#include <conio.h>
#include <windows.h>
#include <mmsystem.h>   // fr WAVEFORMATEX usw.
#include "riff.h"

// Kommandostrings
#define CMD_LADE		"lade"
#define CMD_LIES		"lies"
#define CMD_SCHREIBE	"schreibe"
#define CMD_WAV			"wav"
#define CMD_WAVCUE		"wavcue"
#define CMD_WAVPLAY		"wavplay"
#define CMD_ZEILE		"zeile"
#define CMD_LISTE		"liste"
#define CMD_GERAET		"gert"
#define CMD_HILFE		"hilfe"
#define CMD_ENDE		"ende"

// Lngen der Kommandostrings
#define SIZE_LADE		(sizeof(CMD_LADE) - 1)
#define SIZE_LIES		(sizeof(CMD_LIES) - 1)
#define SIZE_SCHREIBE	(sizeof(CMD_SCHREIBE) - 1)
#define SIZE_WAV		(sizeof(CMD_WAV) - 1)
#define SIZE_WAVCUE		(sizeof(CMD_WAVCUE) - 1)
#define SIZE_WAVPLAY	(sizeof(CMD_WAVPLAY) - 1)
#define SIZE_ZEILE		(sizeof(CMD_ZEILE) - 1)
#define SIZE_LISTE		(sizeof(CMD_LISTE) - 1)
#define SIZE_GERAET		(sizeof(CMD_GERAET) - 1)
#define SIZE_HILFE		(sizeof(CMD_HILFE) - 1)
#define SIZE_ENDE		(sizeof(CMD_ENDE) - 1)

// Dateinamen-Strings
char g_szPhonemFile[256] = "phoneme.wav";
char g_szTextFile[256] = "";
char g_szWavFile[256] = "";

// Zeilenbereich fr Vorlese-Funktion
char g_szLineRange[16] = "";
int g_nLineFrom = 0;
int g_nLineTo = 0;

// Liste der vorhandenen Audio-Wiedergabegerte
#define MAX_DEVICE_COUNT 10
char g_szDeviceList[MAX_DEVICE_COUNT][256];
int g_nDeviceCount = 0;
int g_nDeviceId = WAVE_MAPPER;

// Ein Cue-Element im Cue-Array:
typedef struct _CUE
{
	int nStartSample;             // Erstes Sample innerhalb der Wave-Daten
	int nSampleCount;             // Anzahl Samples fr dieses Segment
	DWORD dwIdentifier;           // Nummer der Cue
	char szLabel[256];
	char szDescription[256];
	char szType[5];
	int nLabelLength;
	DWORD dwOriginalIdentifier;   // Fr g_pCueArrayAusgabe: Ursprngliche Nummer der Cue
} CUE;

CUE *g_pCueArray;             // Das Cue-Array wird dynamisch allokiert.
CUE *g_pCueArrayAusgabe;

// Anzahl Cues (Phoneme)
int g_nCueCount;
int g_nCueCountAusgabe;

// Maximale tatschliche Lnge eines Phonems
int g_nMaxPhonemLen = 0;

// WAVE-Audiodaten
char* g_pWaveData;
int g_nWaveDataSize;
char* g_pWaveDataAusgabe;

// WAVE-Header
HWAVEOUT g_hWaveOut;
WAVEFORMATEX g_WaveFmt;
WAVEHDR g_WaveHdr = { 0, 0, 0, 0, 0, 0, 0, 0 };

HANDLE g_hWaveOutEvent = NULL;

// Optionen
bool g_bKeepDeviceOpen = false;

//
// Makro fr sicheren strcpy()-Ersatz
//
#define StringCopy(pszDest, pszSource) strncpy(memset(pszDest, 0, sizeof(pszDest)), pszSource, sizeof(pszDest) - 1)

//
// Makro zum Kopieren eines String-Arguments mit Standard-Dateierweiterung
//
#define CopyArgument(pszDest, pszSource, pszDefaultFileExt) \
{ \
	char* p = pszSource; \
	while (*p && *p == ' ') p++; \
	if(*p) \
	{ \
		StringCopy(pszDest, p); \
		if(*pszDefaultFileExt && !strrchr(pszDest, '.')) \
			strncat(pszDest, pszDefaultFileExt, sizeof(pszDest) - strlen(pszDest) - 1); \
	} \
}

//
// Dieses Makro setzt pszDest auf basename(pszSource) + pszNewExtension, falls pszDest leer ist.
//
#define SetFilenameIfEmpty(pszDest, pszSource, pszNewExtension) \
{ \
	if(!*pszDest) \
	{ \
		StringCopy(pszDest, pszSource); \
		char *p = strrchr(pszDest, '.'); \
		if(p && strlen(pszDest) + strlen(pszNewExtension) < sizeof(pszDest)) strcpy(p, pszNewExtension); \
	} \
}

bool IsWindowsVistaOrGreater()
{
	OSVERSIONINFOEXW osvi = { sizeof(osvi), 0, 0, 0, 0, {0}, 0, 0 };
	DWORDLONG const dwlConditionMask = VerSetConditionMask
	(
		VerSetConditionMask
		(
			VerSetConditionMask(0, VER_MAJORVERSION, VER_GREATER_EQUAL),
			VER_MINORVERSION, 
			VER_GREATER_EQUAL
		),
		VER_SERVICEPACKMAJOR,
		VER_GREATER_EQUAL
	);
	osvi.dwMajorVersion = HIBYTE(_WIN32_WINNT_VISTA);
	osvi.dwMinorVersion = LOBYTE(_WIN32_WINNT_VISTA);
	osvi.wServicePackMajor = 0;

	return VerifyVersionInfoW(&osvi, VER_MAJORVERSION | VER_MINORVERSION | VER_SERVICEPACKMAJOR, dwlConditionMask) != FALSE;
}

void GetPhonemFilePath()
{
	char szProgramPath[sizeof(g_szPhonemFile)];

	GetModuleFileName(NULL, szProgramPath, sizeof(szProgramPath));
	szProgramPath[sizeof(szProgramPath) - 1] = '\0';
	char *pchBackSlash = strrchr(szProgramPath, '\\');
	if(pchBackSlash != NULL) *(pchBackSlash + 1) = '\0';
	if(strlen(szProgramPath) + strlen(g_szPhonemFile) < sizeof(g_szPhonemFile))
	{
		strcat(szProgramPath, g_szPhonemFile);
		strcpy(g_szPhonemFile, szProgramPath);
	}
}

//
// Parst einen String im Format "n-m" und setzt g_nLineFrom und g_nLineTo entsprechend.
//
void ParseLineRange()
{
	if(*g_szLineRange)
	{
		g_nLineFrom = 0;
		g_nLineTo = 0;

		char* pchMinus = strrchr(g_szLineRange, '-');
		if(pchMinus)
		{
			*pchMinus = '\0';
			sscanf(pchMinus + 1, "%d", &g_nLineTo);
		}
		else g_nLineTo = -1;
		sscanf(g_szLineRange, "%d", &g_nLineFrom);

		if(g_nLineFrom < 0) g_nLineFrom = 0;
		if(g_nLineTo < 0) g_nLineTo = g_nLineFrom;
		if(g_nLineTo && g_nLineFrom > g_nLineTo) g_nLineTo = g_nLineFrom;
	}

	char szLineFrom[16] = "erste";
	if(g_nLineFrom) sprintf(szLineFrom, "%d", g_nLineFrom);

	char szLineTo[16] = "letzte";
	if(g_nLineTo) sprintf(szLineTo, "%d", g_nLineTo);

	*g_szLineRange = '\0'; // String nach dem Parsen leeren, da eh kaputt.

	printf("Zeilenbereich: %s bis %s\n", szLineFrom, szLineTo);
}

//
// Zurcksetzen diverserer Parameter, nachdem eine neue Textdatei geladen wurde.
//
void ResetTextfileParameters()
{
	static char szTextFile[sizeof(g_szTextFile)] = "";

	if(strcmp(szTextFile, g_szTextFile))
	{
		*g_szWavFile = '\0';
		g_nLineFrom = 0;
		g_nLineTo = 0;

		StringCopy(szTextFile, g_szTextFile);
	}
}

void InitDeviceList()
{
	// Die Namen der Audio-Wiedergabegerte in das Array schreiben:

	WAVEOUTCAPS wocaps;
	g_nDeviceCount = waveOutGetNumDevs();
	for(int nIndex = 0; nIndex < g_nDeviceCount && nIndex < MAX_DEVICE_COUNT; nIndex++)
	{
		if(waveOutGetDevCaps(nIndex, &wocaps, sizeof(WAVEOUTCAPS)) == MMSYSERR_NOERROR)
		{
			StringCopy(g_szDeviceList[nIndex], wocaps.szPname);
		}
	}
}

char* GetDeviceName(int nDeviceId)
{
	return nDeviceId == -1 ? "WAVE_MAPPER" : nDeviceId < g_nDeviceCount ? g_szDeviceList[nDeviceId] : "(ungueltig)";
}

void ListDevices()
{
	for(int nIndex = -1; nIndex < g_nDeviceCount; nIndex++)
	{
		printf("%s %3d - %s\n", nIndex == g_nDeviceId ? ">" : " ", nIndex, GetDeviceName(nIndex));
	}
}

void PrintWaveOutError(MMRESULT nCode)
{
  char szErrorText[256];
  waveOutGetErrorText(nCode, szErrorText, sizeof(szErrorText));
  printf("Fehler: %s\n", szErrorText);
}

void PrintLastError()
{
	char *pszErrorText = NULL;

	int nLenErrorText = FormatMessage
	(
		FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_IGNORE_INSERTS | FORMAT_MESSAGE_FROM_SYSTEM,
		NULL,
		GetLastError(),
		MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
		(LPTSTR)&pszErrorText,
		1,
		NULL
	);

	if(nLenErrorText == 0)
		printf("Unbekannter Fehler.\n");
	else
		printf("Fehler: %s\n", pszErrorText);

	LocalFree(pszErrorText);
}

//
// Sicherer gets()-Ersatz
//
char* GetString(char* pszLine, int nSize)
{
	char *pszResult = fgets(pszLine, nSize, stdin);
	if(pszResult)
	{
		int nLen = strlen(pszResult);
		if(nLen > 0 && pszResult[nLen - 1] == '\n')
			pszResult[nLen - 1] = '\0';
	}
	
	return pszResult;
}

bool OpenAudioDevice()
{
	g_hWaveOutEvent = CreateEvent(NULL, FALSE, FALSE, NULL);
	MMRESULT nResult = waveOutOpen(&g_hWaveOut, g_nDeviceId, &g_WaveFmt, (DWORD_PTR)g_hWaveOutEvent, 0, CALLBACK_EVENT);
	if (nResult != MMSYSERR_NOERROR)
	{
		CloseHandle(g_hWaveOutEvent);
		PrintWaveOutError(nResult);
	}
	return nResult == MMSYSERR_NOERROR;
}

void CloseAudioDevice()
{
	waveOutClose(g_hWaveOut);
	CloseHandle(g_hWaveOutEvent);
}

//
// Warten, bis die Wiedergabe beendet ist.
// Ein Tastendruck beendet die Wiedergabe vorzeitig.
//
void WaitForWaveOutDone()
{
	while(1)
	{
		if(_kbhit())
		{
			// _getch(); -> Der Tastendruck bleibt im Tastaturpuffer fr PlayTextFile()!
			waveOutReset(g_hWaveOut);
			return;
		}

		DWORD dwResult = WaitForSingleObject(g_hWaveOutEvent, 0);
		if(dwResult == WAIT_FAILED)
		{
			PrintLastError();
			return;
		}
		else if(g_WaveHdr.dwFlags & WHDR_DONE)
		{
			return;
		}

		Sleep(1); // Etwas Leerlauf, sonst volle Auslastung des CPU-Kerns!
	}
}

//
// Wiedergabe eines Wave-Datenpuffers.
// Das Audio-Wiedergabegert muss bereits geffnet sein!
//
bool Play(char* pWaveData, int nWaveDataSize)
{
	if(!g_bKeepDeviceOpen && !OpenAudioDevice()) return false;
	g_WaveHdr.lpData = pWaveData;
	g_WaveHdr.dwBufferLength = nWaveDataSize;
	waveOutPrepareHeader(g_hWaveOut, &g_WaveHdr, sizeof(WAVEHDR));
	waveOutWrite(g_hWaveOut, &g_WaveHdr, sizeof(WAVEHDR));
	WaitForWaveOutDone();
	waveOutUnprepareHeader(g_hWaveOut, &g_WaveHdr, sizeof(WAVEHDR));
	if(!g_bKeepDeviceOpen) CloseAudioDevice();
	return true;
}

//
// Funktionen zum Sortieren des Cue-Arrays:
//

void CueArrayElementToSortString(CUE* pCueArrayElement, char* pszString)
{
	sprintf
	(
		pszString,
		"%010d%010d%s%s%s",
		pCueArrayElement->nStartSample,
		pCueArrayElement->nSampleCount,
		pCueArrayElement->szLabel,
		pCueArrayElement->szDescription,
		pCueArrayElement->szType
	);
}

int CompareCueArrayElements(CUE* pElement1, CUE* pElement2)
{
	char szElement1[10 + 10 + sizeof(pElement1[0].szLabel) + sizeof(pElement1[0].szDescription) + sizeof(pElement1[0].szType) + 1];
	char szElement2[10 + 10 + sizeof(pElement2[0].szLabel) + sizeof(pElement2[0].szDescription) + sizeof(pElement2[0].szType) + 1];

	CueArrayElementToSortString(pElement1, szElement1);
	CueArrayElementToSortString(pElement2, szElement2);

	return strcmp(szElement1, szElement2);
}

void SwapCueArrayElements(CUE* pElement1, CUE* pElement2)
{
	CUE tmpCueArrayElement;

	memcpy(&tmpCueArrayElement, pElement1, sizeof(CUE));
	memcpy(pElement1, pElement2, sizeof(CUE));
	memcpy(pElement2, &tmpCueArrayElement, sizeof(CUE));
}

void SortCueArray(CUE* pCueArray)
{
	if(g_nCueCount < 2) return;

	// Bubblesort
	for(int iEnde = g_nCueCount - 1; iEnde >= 0 ; iEnde--)
	{
		for(int iElement = 0; iElement < iEnde; iElement++)
		{
			if(CompareCueArrayElements(&pCueArray[iElement], &pCueArray[iElement + 1]) > 0)
			{
				SwapCueArrayElements(&pCueArray[iElement], &pCueArray[iElement + 1]);
			}
		}
	}
}

//
// Ldt die WAV-Datei mit den Phonemen.
// Die enthaltene Cue-List enthlt die Metainformationen der einzelnen Phoneme. 
//
bool LoadWavFile()
{
	MMCKINFO RiffChunkInfo; // RIFF-Chunk
	MMCKINFO FormatChunkInfo; // Format-Subchunk (Wave-Parameter)
	MMCKINFO DataChunkInfo; // Data-Subchunk (Wave-Audio-Daten)
	MMCKINFO CueChunkInfo; // Cue-Subchunk (Startsample)
	MMCKINFO ListChunkInfo; // LIST-Chunk
	MMCKINFO LtxtChunkInfo; // Cue-Lnge
	MMCKINFO LablChunkInfo; // Cue-Label
	MMCKINFO NoteChunkInfo; // Cue-Description
	CUEPOINT CuePoint;
	CUELABELTEXTCHUNK CueLtxtChunk;
	CUETEXTCHUNK CueTextChunk;

	FILE *pWavFile = fopen(g_szPhonemFile, "rb");
	if(!pWavFile)
	{
		printf("Fehler: Die Phonemdatei \"%s\" kann nicht geffnet werden.\n", g_szPhonemFile);
		return false;
	}

	// Find the RIFF chunk.
	memset(&RiffChunkInfo, 0, sizeof(MMCKINFO));
	RiffChunkInfo.ckid = StringToFOURCC("RIFF");
	if(FindChunk(pWavFile, &RiffChunkInfo, NULL))
	{
		fclose(pWavFile);
		printf("Fehler: Der RIFF-Chunk konnte in der Phonemdatei \"%s\" nicht gefunden werden.\n", g_szPhonemFile);
		return false;
	}

	// Descend into the format chunk.
	memset(&FormatChunkInfo, 0, sizeof(MMCKINFO));
	FormatChunkInfo.ckid = StringToFOURCC("fmt");
	if(FindChunk(pWavFile, &FormatChunkInfo, &RiffChunkInfo))
	{
		fclose(pWavFile);
		printf("Fehler: Der fmt-Chunk konnte in der Phonemdatei \"%s\" nicht gefunden werden.\n", g_szPhonemFile);
		return false;
	}

	// Read the wave format.
	memset(&g_WaveFmt, 0, sizeof(WAVEFORMATEX));
	int nBytesToRead = sizeof(WAVEFORMATEX) < FormatChunkInfo.cksize ? sizeof(WAVEFORMATEX) : FormatChunkInfo.cksize;
	fread((char*)&g_WaveFmt, 1, nBytesToRead, pWavFile);

	// Ascend out of the format chunk.
	if(SeekBehindChunk(pWavFile, &FormatChunkInfo))
	{
		fclose(pWavFile);
		printf("Fehler: Der fmt-Chunk konnte in der Phonemdatei \"%s\" nicht gefunden werden.\n", g_szPhonemFile);
		return false;
	}

	// Descend into the data chunk.
	memset(&DataChunkInfo, 0, sizeof(MMCKINFO));
	DataChunkInfo.ckid = StringToFOURCC("data");
	if(FindChunk(pWavFile, &DataChunkInfo, &RiffChunkInfo))
	{
		fclose(pWavFile);
		printf("Fehler: Der data-Chunk konnte in der Phonemdatei \"%s\" nicht gefunden werden.\n", g_szPhonemFile);
		return false;
	}

	// PCM-Daten einlesen
	g_nWaveDataSize = DataChunkInfo.cksize;
	g_pWaveData = malloc(g_nWaveDataSize);
	fread(g_pWaveData, 1, g_nWaveDataSize, pWavFile);

	// Cue-List ins Array einlesen:
	do
	{
		// Ascend out of the data chunk.
		if(SeekBehindChunk(pWavFile, &DataChunkInfo)) break;

		// Descend into the cue chunk.
		memset(&CueChunkInfo, 0, sizeof(MMCKINFO));
		CueChunkInfo.ckid = StringToFOURCC("cue");
		if(FindChunk(pWavFile, &CueChunkInfo, &RiffChunkInfo)) break;

		// Cue-Positions lesen
		fread((char*)&g_nCueCount, sizeof(DWORD), 1, pWavFile); // Anzahl Cuepoints
		if(g_nCueCount == 0) break; // Keine Cues gefunden

		if(sizeof(DWORD) + g_nCueCount * sizeof(CUEPOINT) != CueChunkInfo.cksize)
		{
			printf("Fehler: Die Anzahl der Cues in der Phonemdatei \"%s\" passt nicht zur Gre des Cue-Chunks.\n", g_szPhonemFile);
			g_nCueCount = 0;
			break;
		}

		// Cue-Array allokieren
		g_pCueArray = malloc(sizeof(CUE) * g_nCueCount);

		// Cue-Positions in das Cue-Array eintragen
		for(int i = 0; i < g_nCueCount; i++)
		{
			memset(&CuePoint, 0, sizeof(CUEPOINT));
			fread((char*)&CuePoint, sizeof(CUEPOINT), 1, pWavFile); // Cuepoints in Struktur einlesen
			g_pCueArray[i].nStartSample = CuePoint.dwSampleOffset;
			g_pCueArray[i].dwIdentifier = CuePoint.dwIdentifier;
			g_pCueArray[i].nSampleCount = 0;
			g_pCueArray[i].szLabel[0] = '\0';
			g_pCueArray[i].szDescription[0] = '\0';
			g_pCueArray[i].szType[0] = '\0';
		}

		// Ascend out of the cue chunk.
		if(SeekBehindChunk(pWavFile, &CueChunkInfo)) break;

		// Find the LIST chunk.
		memset(&ListChunkInfo, 0, sizeof(MMCKINFO));
		ListChunkInfo.ckid = StringToFOURCC("LIST");
		if(FindChunk(pWavFile, &ListChunkInfo, NULL))
		{
			g_nCueCount = 0;
			break;
		}

		do // Cue-Lngen lesen:
		{
			// Descend into the ltxt chunk.
			memset(&LtxtChunkInfo, 0, sizeof(MMCKINFO));
			LtxtChunkInfo.ckid = StringToFOURCC("ltxt");
			if(FindChunk(pWavFile, &LtxtChunkInfo, &ListChunkInfo)) break;

			memset(&CueLtxtChunk, 0, sizeof(CUELABELTEXTCHUNK));
			fread((char*)&CueLtxtChunk + 8, LtxtChunkInfo.cksize, 1, pWavFile);

			for(int i = 0; i < g_nCueCount; i++)
			{
				if(g_pCueArray[i].dwIdentifier > 0 && g_pCueArray[i].dwIdentifier == CueLtxtChunk.dwIdentifier)
				{
					g_pCueArray[i].nSampleCount = CueLtxtChunk.dwSampleLength;
					FOURCCToString(CueLtxtChunk.dwPurpose, g_pCueArray[i].szType);
					if(g_pCueArray[i].szType[0] == '\0') StringCopy(g_pCueArray[i].szType, "rgn");
					break;
				}
			}
			// Ascend out of the ltxt chunk.
			if(SeekBehindChunk(pWavFile, &LtxtChunkInfo)) break;
		}
		while(true);

		fseek(pWavFile, ListChunkInfo.dwDataOffset + 4, SEEK_SET); // Zum Beginn des List-Chunks!

		do // Cue-Labels lesen:
		{
			// Descend into the labl chunk.
			memset(&LablChunkInfo, 0, sizeof(MMCKINFO));
			LablChunkInfo.ckid = StringToFOURCC("labl");
			if(FindChunk(pWavFile, &LablChunkInfo, &ListChunkInfo)) break;

			memset(&CueTextChunk, 0, sizeof(CUETEXTCHUNK));
			fread((char*)&CueTextChunk + 8, LablChunkInfo.cksize, 1, pWavFile);

			for(int i = 0; i < g_nCueCount; i++)
			{
				if(g_pCueArray[i].dwIdentifier > 0 && g_pCueArray[i].dwIdentifier == CueTextChunk.dwIdentifier)
				{
					StringCopy(g_pCueArray[i].szLabel, CueTextChunk.szText);
					g_pCueArray[i].nLabelLength = strlen(g_pCueArray[i].szLabel);
					g_nMaxPhonemLen = max(g_nMaxPhonemLen, g_pCueArray[i].nLabelLength);
					break;
				}
			}
			// Ascend out of the labl chunk.
			if(SeekBehindChunk(pWavFile, &LablChunkInfo)) break;
		}
		while(true);

		fseek(pWavFile, ListChunkInfo.dwDataOffset + 4, SEEK_SET); // Zum Beginn des List-Chunks!

		do // Cue-Descriptions lesen:
		{
			// Descend into the note chunk.
			memset(&NoteChunkInfo, 0, sizeof(MMCKINFO));
			NoteChunkInfo.ckid = StringToFOURCC("note");
			if(FindChunk(pWavFile, &NoteChunkInfo, &ListChunkInfo)) break;

			memset(&CueTextChunk, 0, sizeof(CUETEXTCHUNK));
			fread((char*)&CueTextChunk + 8, NoteChunkInfo.cksize, 1, pWavFile);

			for(int i = 0; i < g_nCueCount; i++)
			{
				if(g_pCueArray[i].dwIdentifier > 0 && g_pCueArray[i].dwIdentifier == CueTextChunk.dwIdentifier)
				{
					StringCopy(g_pCueArray[i].szDescription, CueTextChunk.szText);
					break;
				}
			}
			// Ascend out of the labl chunk.
			if(SeekBehindChunk(pWavFile, &NoteChunkInfo)) break;
		}
		while(true);
	}
	while(false); // "Schleife" nur einmal durchlaufen

	// Close the file
	fclose(pWavFile);

	if(g_WaveFmt.wFormatTag != WAVE_FORMAT_PCM || (g_WaveFmt.wBitsPerSample != 8 && g_WaveFmt.wBitsPerSample != 16) || (g_WaveFmt.nChannels != 1 && g_WaveFmt.nChannels != 2))
	{
		printf("Fehler: Die Phonemdatei \"%s\" ist keine Standard-PCM-Datei (mono oder stereo, 8 oder 16 Bits/Sample).\n", g_szPhonemFile);
		return false;
	}

	return true;
}

//
// Schreibt eine WAV-Datei mit Cue-List.
// Die Cue-List enthlt die Metainformationen der einzelnen Phoneme. 
//
bool WriteWavFile(FILE *pWavFile, bool bWriteCueList, bool bWritePlayList)
{
	bool bSuccess = false;

	MMCKINFO RiffChunkInfo; // RIFF-Chunk
	MMCKINFO SubChunkInfo;
	MMCKINFO ListChunkInfo; // LIST-Chunk
	CUECHUNK CueChunkInfo; // Cue-Subchunk (Startsample)
	CUEPOINT CuePointInfo;
	CUELABELTEXTCHUNK CueLabelTextChunkInfo;
	CUETEXTCHUNK CueTextChunkInfo;
	PLAYLISTCHUNK PlayListChunk;
	PLAYLISTITEM PlayListItem;

	int nDataSize = 0;
	int nSampleCount = 0;

	bool bError = false;

	do
	{
		// RIFF-Chunk schreiben (cksize noch uninitialisiert!)
		memset(&RiffChunkInfo, 0, sizeof(MMCKINFO));
		RiffChunkInfo.ckid = StringToFOURCC("RIFF");
		RiffChunkInfo.fccType = StringToFOURCC("WAVE");
		if(!fwrite(&RiffChunkInfo, 12, 1, pWavFile)) break;

		// Format-Chunk schreiben
		memset(&SubChunkInfo, 0, sizeof(MMCKINFO));
		SubChunkInfo.ckid = StringToFOURCC("fmt");
		SubChunkInfo.cksize = 16;
		if(!fwrite(&SubChunkInfo, 8, 1, pWavFile)) break;

		// WAVE-Format schreiben
		if(!fwrite(&g_WaveFmt, SubChunkInfo.cksize, 1, pWavFile)) break;

		// Data-Subchunk schreiben (cksize noch uninitialisiert!)
		int nDataPos = ftell(pWavFile);
		SubChunkInfo.ckid = StringToFOURCC("data");
		if(!fwrite(&SubChunkInfo, 8, 1, pWavFile)) break;

		// WAVE-Daten schreiben
		if(bWritePlayList)
		{
			// Bei Play-List die Wave-Daten der Phonemdatei schreiben:
			if(!fwrite(g_pWaveData, g_nWaveDataSize, 1, pWavFile)) break;
			nDataSize += g_nWaveDataSize;
		}
		else
		{
			// Ansonsten die Wave-Datenblcke der Ausgabe-Cue-List schreiben:
			int nBytesPerSample = (g_WaveFmt.wBitsPerSample / 8) * g_WaveFmt.nChannels;
			for(int i = 0; i < g_nCueCountAusgabe; i++)
			{
				int nWaveOffset = g_pCueArrayAusgabe[i].nStartSample * nBytesPerSample;
				int nWaveSize = g_pCueArrayAusgabe[i].nSampleCount * nBytesPerSample;

				if(!fwrite(g_pWaveData + nWaveOffset, nWaveSize, 1, pWavFile))
				{
					bError = true;
					break;
				}
				nDataSize += nWaveSize;

				// Startsample von g_pWaveData auf Datei umsetzen:
				g_pCueArrayAusgabe[i].nStartSample = nSampleCount;
				nSampleCount += g_pCueArrayAusgabe[i].nSampleCount;
			}
			if(bError) break;
		}

		// Falls WAVE-Daten-Gre ungerade, mit einem Nullbyte auffllen,
		// damit wird die Gre des RIFF-Chunks gerade wird:
		if(nDataSize % 2)
		{
			char chPadByte = '\0';
			if(!fwrite(&chPadByte, 1, 1, pWavFile)) break;
		}

		int nListPos = 0, nListSize = 0; // Position und Gre des LIST-Chunks

		// Das richtige Cue-Array whlen
		CUE *pCueArray = bWritePlayList ? g_pCueArray : g_pCueArrayAusgabe;
		int nCueCount = bWritePlayList ? g_nCueCount : g_nCueCountAusgabe;

		if(bWriteCueList || bWritePlayList)
		{
			// Cue-Subchunk schreiben
			memset(&CueChunkInfo, 0, sizeof(CUECHUNK));
			CueChunkInfo.chunkID = StringToFOURCC("cue");
			CueChunkInfo.chunkSize = 4 + nCueCount * sizeof(CUEPOINT);
			CueChunkInfo.dwCuePoints = nCueCount;
			if(!fwrite(&CueChunkInfo, sizeof(CUECHUNK), 1, pWavFile)) break;

			// Cue-Position-Strukturen schreiben
			for(int i = 0; i < nCueCount; i++)
			{
				memset(&CuePointInfo, 0, sizeof(CUEPOINT));
				CuePointInfo.dwIdentifier = pCueArray[i].dwIdentifier;
				CuePointInfo.dwPosition = pCueArray[i].nStartSample;
				CuePointInfo.fccChunk = StringToFOURCC("data");
				CuePointInfo.dwChunkStart = 0;
				CuePointInfo.dwBlockStart = 0;
				CuePointInfo.dwSampleOffset = pCueArray[i].nStartSample;;

				if(!fwrite(&CuePointInfo, sizeof(CUEPOINT), 1, pWavFile))
				{
					bError = true;
					break;
				}
			}
			if(bError) break;

			// LIST-Chunk schreiben (cksize noch uninitialisiert!)
			nListPos = ftell(pWavFile);
			memset(&ListChunkInfo, 0, sizeof(MMCKINFO));
			ListChunkInfo.ckid = StringToFOURCC("LIST");
			ListChunkInfo.fccType = StringToFOURCC("adtl");
			if(!fwrite(&ListChunkInfo, 12, 1, pWavFile)) break;

			// Nun die Chunks fr Cue-Lnge, -Label und Description schreiben
			
			// Fr jede Cue die Chunks schreiben
			nListSize = 4; // fccType
			for(int i = 0; i < nCueCount; i++)
			{
				// CUELABELTEXTCHUNK ("ltxt")

				CueLabelTextChunkInfo.chunkID = StringToFOURCC("ltxt");
				CueLabelTextChunkInfo.chunkSize = 20;
				CueLabelTextChunkInfo.dwIdentifier = pCueArray[i].dwIdentifier;
				CueLabelTextChunkInfo.dwSampleLength = pCueArray[i].nSampleCount;
				CueLabelTextChunkInfo.dwPurpose = StringToFOURCC(pCueArray[i].szType);
				CueLabelTextChunkInfo.dw1 = 0;
				CueLabelTextChunkInfo.dw2 = 0;

				if(!fwrite(&CueLabelTextChunkInfo, sizeof(CUELABELTEXTCHUNK), 1, pWavFile))
				{
					bError = true;
					break;
				}

				nListSize += sizeof(CUELABELTEXTCHUNK);

				// CUETEXTCHUNK ("labl")

				memset(&CueTextChunkInfo, 0, sizeof(CUETEXTCHUNK));

				int nCueTextSize = strlen(pCueArray[i].szLabel) + 1; // Lnge incl. abschlieender 0
				if(nCueTextSize % 2) pCueArray[i].szLabel[nCueTextSize++] = '\0'; // falls ungerade, mit einer weiteren 0 auffllen

				CueTextChunkInfo.chunkID = StringToFOURCC("labl");
				CueTextChunkInfo.chunkSize = 4 + nCueTextSize;
				CueTextChunkInfo.dwIdentifier = pCueArray[i].dwIdentifier;
				StringCopy(CueTextChunkInfo.szText, pCueArray[i].szLabel);

				if(!fwrite(&CueTextChunkInfo, 12 + nCueTextSize, 1, pWavFile))
				{
					bError = true;
					break;
				}

				nListSize += (12 + nCueTextSize);

				// CUETEXTCHUNK ("note")

				nCueTextSize = strlen(pCueArray[i].szDescription) + 1; // Lnge incl. abschlieender 0
				if(nCueTextSize % 2) pCueArray[i].szDescription[nCueTextSize++] = '\0'; // falls ungerade, mit einer weiteren 0 auffllen

				CueTextChunkInfo.chunkID = StringToFOURCC("note");
				CueTextChunkInfo.chunkSize = 4 + nCueTextSize;
				CueTextChunkInfo.dwIdentifier = pCueArray[i].dwIdentifier;
				StringCopy(CueTextChunkInfo.szText, pCueArray[i].szDescription);

				if(!fwrite(&CueTextChunkInfo, 12 + nCueTextSize, 1, pWavFile))
				{
					bError = true;
					break;
				}

				nListSize += (12 + nCueTextSize);
			}
			if(bError) break;
		}

		if(bWritePlayList)
		{
			// PlayList-Chunk schreiben
			memset(&PlayListChunk, 0, sizeof(PLAYLISTCHUNK));
			PlayListChunk.chunkID = StringToFOURCC("plst");
			PlayListChunk.chunkSize = 4 + g_nCueCountAusgabe * sizeof(PLAYLISTITEM);
			PlayListChunk.dwSegments = g_nCueCountAusgabe;
			if(!fwrite(&PlayListChunk, sizeof(PLAYLISTCHUNK), 1, pWavFile)) break;

			// PlayList-Item-Strukturen schreiben
			for(int i = 0; i < g_nCueCountAusgabe; i++)
			{
				// PLAYLISTITEM PlayListItem
				memset(&PlayListItem, 0, sizeof(PLAYLISTITEM));
				PlayListItem.dwIdentifier = g_pCueArrayAusgabe[i].dwOriginalIdentifier;
				PlayListItem.dwLength = g_pCueArrayAusgabe[i].nSampleCount;
				PlayListItem.dwRepeats = 1;

				if(!fwrite(&PlayListItem, sizeof(PLAYLISTITEM), 1, pWavFile))
				{
					bError = true;
					break;
				}
			}
			if(bError) break;
		}

		// cksize im RIFF-Chunk aktualisieren
		int nRiffSize = ftell(pWavFile) - 8;
		if(fseek(pWavFile, 4, SEEK_SET)) break;
		if(!fwrite(&nRiffSize, 4, 1, pWavFile)) break;

		// cksize im Data-Subchunk aktualisieren
		if(fseek(pWavFile, nDataPos + 4, SEEK_SET)) break;
		if(!fwrite(&nDataSize, 4, 1, pWavFile)) break;

		if(bWriteCueList || bWritePlayList)
		{
			// cksize im LIST-Chunk aktualisieren
			if(fseek(pWavFile, nListPos + 4, SEEK_SET)) break;
			if(!fwrite(&nListSize, 4, 1, pWavFile)) break;
		}
		
		bSuccess = true;
	}
	while(false);

	return bSuccess;
}

//
// Sucht das Phonem an der angegebenen Zeichenposition im Phonem-Array
// und setzt die Zeichenposition bei Erfolg hinter das gefundene Phonem.
// Die Phoneme mit den lngeren Bezeichnungen werden zuerst durchsucht.
//
int FindPhonem(char** ppchChar)
{
	// Zeilenumbrche berspringen
	while(**ppchChar == '\n')
		(*ppchChar)++;

	for(int nLen = min(strlen(*ppchChar), g_nMaxPhonemLen); nLen > 0; --nLen)
	{
		for(int nCue = 0; nCue < g_nCueCount; nCue++)
		{
			if(g_pCueArray[nCue].nLabelLength == nLen)
			{
				if(strncmp(*ppchChar, g_pCueArray[nCue].szLabel, nLen) == 0)
				{
					(*ppchChar) += nLen;
					return nCue;
				}
			}
		}
	}

	return -1;
}

//
// Spricht den angegebenen Satz aus Phonemen oder schreibt diesen in die angegebene WAV-Datei.
// Bei Audio-Wiedergabe muss das Audio-Wiedergabegert bereits geffnet sein!
//
bool SpeakSentence(char *pszSentence, bool bWriteAudioToWavFile)
{
	bool bSuccess = true;

	int nWaveDataAusgabeSize = 0;
	int nCueCountAusgabe = 0;

	// 2 Durchgnge (passes):
	// - Pass 0: Ermitteln des bentigten Speicherplatzes fr die Wave-Daten des kompletten Satzes
	// - Pass 1: Allokieren des bentigten Speicherplatzes und Kopieren der Wave-Daten des kompletten Satzes
	for(int nPass = 0; nPass < 2; nPass++)
	{
		char *pchChar = pszSentence; // Zeiger auf aktuelles Zeichen innerhalb des Satzes

		// Phonem-Strings innerhalb des Satzes suchen und verarbeiten
		while(*pchChar)
		{
			// Aktuellen Phonem-String im Phonem-Array suchen
			int nCue = FindPhonem(&pchChar); // FindPhonem setzt pchChar nur weiter, wenn gefunden.
			if(nCue == -1)
			{
				// Phonem-String nicht im Phonem-Array gefunden
				if(nPass == 0) printf("Syntaxfehler bei Zeichen %d: \"%.*s\"\n", pchChar - pszSentence + 1, g_nMaxPhonemLen, pchChar);

				// Nchstes Zeichen
				pchChar++;
			}
			else
			{
				// Zugehrige Wave-Audiodaten ermitteln
				int nBytesPerSample = (g_WaveFmt.wBitsPerSample / 8) * g_WaveFmt.nChannels;
				int nWaveDataCueSize = g_pCueArray[nCue].nSampleCount * nBytesPerSample;

				if(bWriteAudioToWavFile)
				{
					if(nPass == 1)
					{
						// Erst im 2. Durchlauf (Pass 1) die Cue-Daten der auszugebenden Phoneme ins Ausgabe-Cue-Array schreiben:
						g_pCueArrayAusgabe[g_nCueCountAusgabe].nStartSample = g_pCueArray[nCue].nStartSample;
						g_pCueArrayAusgabe[g_nCueCountAusgabe].nSampleCount = g_pCueArray[nCue].nSampleCount;
						g_pCueArrayAusgabe[g_nCueCountAusgabe].dwIdentifier = g_nCueCountAusgabe + 1;
						g_pCueArrayAusgabe[g_nCueCountAusgabe].dwOriginalIdentifier = g_pCueArray[nCue].dwIdentifier;
						StringCopy(g_pCueArrayAusgabe[g_nCueCountAusgabe].szLabel, g_pCueArray[nCue].szLabel);
						StringCopy(g_pCueArrayAusgabe[g_nCueCountAusgabe].szDescription, g_pCueArray[nCue].szDescription);
						StringCopy(g_pCueArrayAusgabe[g_nCueCountAusgabe].szType, g_pCueArray[nCue].szType);
						g_nCueCountAusgabe++;
					}
					nCueCountAusgabe++;
				}
				else
				{
					if(nPass == 1)
					{
						// Erst im 2. Durchlauf (Pass 1) die Wave-Audiodaten des aktuellen Phonems in den Ausgabepuffer schreiben:
						char* pWaveDataCueStart = g_pWaveData + g_pCueArray[nCue].nStartSample * nBytesPerSample;
						memcpy(g_pWaveDataAusgabe + nWaveDataAusgabeSize, pWaveDataCueStart, nWaveDataCueSize);
					}
					nWaveDataAusgabeSize += nWaveDataCueSize;
				}
			}
		}

		if(nPass == 0)
		{
			// Nach Beendigung des 1. Durchlaufs (Pass 0) ...
			if(bWriteAudioToWavFile)
			{
				// ... Ausgabe-Cue-Array re-allokieren:
				g_pCueArrayAusgabe = realloc(g_pCueArrayAusgabe, sizeof(CUE) * (g_nCueCountAusgabe + nCueCountAusgabe));
				nCueCountAusgabe = 0;
			}
			else
			{
				// ... den ermittelten Speicherplatz fr die Wave-Daten des kompletten Satzes allokieren:
				g_pWaveDataAusgabe = malloc(nWaveDataAusgabeSize);
				nWaveDataAusgabeSize = 0;
			}
		}
	}

	if(!bWriteAudioToWavFile)
	{
		// Andernfalls die Wave-Audiodaten wiedergeben:
		bSuccess = Play(g_pWaveDataAusgabe, nWaveDataAusgabeSize);

		// Nach der Verarbeitung der Wave-Audiodaten den allokierten Speicher wieder freigeben:
		free(g_pWaveDataAusgabe);
		g_pWaveDataAusgabe = NULL;
	}

	return bSuccess;
}

//
// Spricht den angegebenen Satz aus Phonemen.
//
bool Sprich(char *pszSentence)
{
	if(g_bKeepDeviceOpen && !OpenAudioDevice()) return false;
	bool bResult = SpeakSentence(pszSentence, false);
	if(g_bKeepDeviceOpen) CloseAudioDevice();
	return bResult;
}

void PrintHelpText(void)
{
//	Sprich("hilf3tzumprogram");

	char szLineFrom[16] = "";
	if(g_nLineFrom) sprintf(szLineFrom, "%d", g_nLineFrom);

	char szLineTo[16] = "";
	if(g_nLineTo) sprintf(szLineTo, "%d", g_nLineTo);

	char *pszPhonemFileBaseName = strrchr(g_szPhonemFile, '\\');
	pszPhonemFileBaseName = pszPhonemFileBaseName ? pszPhonemFileBaseName + 1 : g_szPhonemFile;

	printf
	(
		"\nSPRICH V1.0  Copyright (c) 1993 Stefan Bion (Build: %s)\n"
		"Sie knnen beim Aufruf des Programmes SPRICH folgende Optionen angeben:\n"
		"\n"
		"  -p<Phonemdatei>[.wav]     Phonemdatei (WAV-Datei mit Cue-List) [%s]\n"
		"  -o0 oder -o1              Audiogert whrend Wiedergabe geffnet halten [%d]\n"
		"  <Textdatei>               Diese Textdatei (Phonemschrift) wird sofort vorgelesen\n"
		"\n"
		"Im interaktiven Modus stehen folgende Kommandos zur Verfgung:\n"
		"\n"
		"  lade [<Datei>[.txt]]      Ldt die Textdatei [%s]\n"
		"  lies [<Datei>[.txt]]      Liest die Textdatei vor\n"
		"  schreibe [<Datei>[.txt]]  Hngt den zuletzt eingegebenen Satz an die Textdatei an\n"
		"  wav [<Datei>[.wav]]       Erzeugt aus der geladenen Textdatei eine WAV-Datei\n"
		"  wavcue [<Datei>[.wav]]    Wie 'wav', jedoch zustzlich mit Cue-Liste der Phoneme\n"
		"  wavplay [<Datei>[.wav]]   Wie 'wav', jedoch komprimiert mit Play-Liste zur Wiedergabe\n"
		"  zeile [von][-][bis]       Setzt Von-/Bis-Zeile fr 'lies' und 'wav...' [%s-%s]\n"
		"  liste                     Listet alle Phoneme mit Lnge und Beschreibung auf\n"
		"  gert [n]                 Auswahl des Audio-Wiedergabegertes [%s]\n"
		"  hilfe                     Zeigt diesen Hilfstext an\n"
		"  ende                      Beendet das Programm SPRICH\n"
		"\n"
		"Ansonsten knnnen Sie direkt Phonemschrift eingeben, die das Programm sofort\n"
		"nachpricht. Eine leere Eingabe (nur RETURN) wiederholt die letzte Wiedergabe.\n\n"
		, __DATE__ ", " __TIME__, pszPhonemFileBaseName, g_bKeepDeviceOpen, g_szTextFile, szLineFrom, szLineTo, GetDeviceName(g_nDeviceId)
	);
}

void PrintPhonemList(void)
{
	printf("Nr. | Dauer [s] | Phonem   | Beschreibung\n");
	printf("----+-----------+----------+----------------------------------------------------\n");

	for(int nCue = 0; nCue < g_nCueCount; nCue++)
	{
		printf
		(
			"%3d | %9.3f | %-8s | %s\n",
			nCue + 1,
			(double)g_pCueArray[nCue].nSampleCount / g_WaveFmt.nSamplesPerSec,
			g_pCueArray[nCue].szLabel,
			g_pCueArray[nCue].szDescription
		);
	}
}

//
// Vorlesen einer Phonemschrift-Textdatei.
// Falls bWriteAudioToWavFile == true ist, werden die
// Audiodaten stattdessen in die zuvor angegebene WAV-Datei geschrieben.
//
bool PlayTextFile(bool bWriteAudioToWavFile, bool bWriteCueList, bool bWritePlayList)
{
	bool bSuccess = true;

	FILE *pTextFile = NULL;
	if((pTextFile = fopen(g_szTextFile, "r")) == NULL)
	{
		printf("Fehler: Die Textdatei \"%s\" kann nicht geffnet werden.\n", g_szTextFile);
		return false;
	}

	FILE *pWavFile = NULL;
	if(bWriteAudioToWavFile)
	{
		g_pCueArrayAusgabe = malloc(sizeof(CUE)); // Wird spter mit realloc() vergrert!
		g_nCueCountAusgabe = 0;

		printf("Schreibe Audiodaten der Textdatei \"%s\" in die WAV-Datei \"%s\" ...\n", g_szTextFile, g_szWavFile);
	}
	else
	{
		// Sound-Device ffnen (dies wird schon hier gemacht und nicht erst in Play(),
		// um die Pausen zwischen der Wiedergabe der einzelnen Stze zu minimieren.)
		if (g_bKeepDeviceOpen && !OpenAudioDevice())
		{
			fclose(pTextFile);
			return false;
		}

		printf("Lese Textdatei \"%s\" vor - zum Beenden beliebige Taste drcken ...\n", g_szTextFile);
	}

	// Textdatei zeilenweise lesen; ein Tastendruck - erfolgt in WaitForWaveOutDone() - beendet die Verarbeitung.
	char szLine[32768];
	for(int nLineNo = 1; bSuccess && fgets(szLine, sizeof(szLine), pTextFile) != NULL && !_kbhit(); nLineNo++)
	{
		// Zeilenbereich bercksichtigen
		if(g_nLineFrom && nLineNo < g_nLineFrom) continue;
		if(g_nLineTo && nLineNo > g_nLineTo) continue;

		szLine[sizeof(szLine) - 1] = '\0';

		// Zeilenumbruch am Zeilenende entfernen
		int nLineLength = strlen(szLine);
		if(nLineLength > 0 && szLine[nLineLength - 1] == '\n')
			szLine[nLineLength - 1] = '\0';

		// Leerzeilen und Kommentarzeilen berspringen
		if(*szLine == 0 || *szLine == '*')
		{
			// Bei Audio-Wiedergabe die aktuelle Leer- oder Kommentarzeile aus der Textdatei anzeigen
			if(!bWriteAudioToWavFile)
				printf("%4d: %s\n", nLineNo, szLine);

			continue;
		}
		else
		{
			// Bei Audio-Wiedergabe den aktuellen Satz aus der Textdatei anzeigen
			if(!bWriteAudioToWavFile)
				printf("%4d: %s\n", nLineNo, szLine);

			// Satz sprechen oder Cue-List fr Ausgabe erzeugen
			bSuccess = SpeakSentence(szLine, bWriteAudioToWavFile);
		}
	}

	// Eventuellen Tastendruck aus dem Tastaturpuffer entfernen
	if(_kbhit())
		_getch();

	// Textdatei schlieen
	fclose(pTextFile);

	if(bWriteAudioToWavFile)
	{
		// WAV-Datei schreiben
		if((pWavFile = fopen(g_szWavFile, "wb")) == NULL)
		{
			printf("Fehler: Die WAV-Datei \"%s\" kann nicht erstellt werden.\n", g_szWavFile);
			bSuccess = false;
		}
		else
		{
			if(!WriteWavFile(pWavFile, bWriteCueList, bWritePlayList))
			{
				printf("Fehler: Die Textdatei \"%s\" kann nicht beschrieben werden.\n", g_szTextFile);
				bSuccess = false;
			}
			fclose(pWavFile);
		}
		free(g_pCueArrayAusgabe);
		g_pCueArrayAusgabe = NULL;
	}
	else
	{
		if(g_bKeepDeviceOpen) CloseAudioDevice();
	}

	return bSuccess;
}

void AppendToTextFile(char *pszLine)
{
	if(!*pszLine)
	{
		printf("Es gibt nichts zu schreiben.\n");
		return;
	}

	FILE *pTextFile = fopen(g_szTextFile, "a");

	if(pTextFile)
	{
		printf("Hnge \"%s\" an Textdatei \"%s\" an.\n", pszLine, g_szTextFile);
		fprintf(pTextFile, "%s\n", pszLine);
		fclose(pTextFile);
	}
	else
	{
		printf("Fehler: Die Textdatei \"%s\" kann nicht beschrieben werden.\n", g_szTextFile);
	}
}

int main(int argc, char *argv[])
{
	// Codepage der Konsole auf Windows-ANSI setzen:
	SetConsoleCP(1252); // fr Tastatureingaben
	SetConsoleOutputCP(1252); // fr Bildschirmausgaben

	// Bei Windows-Versionen ab Vista das Audio-Wiedergabegert whrend der Wiedergabe 
	// der einzelnen Textzeilen geffnet halten, um Pausenzeiten zu minimieren!
	g_bKeepDeviceOpen = IsWindowsVistaOrGreater();

	// Kompletten Pfad der Phonemdatei ermitteln
	GetPhonemFilePath();

	// Kommandozeile parsen
	while(--argc)
	{
		switch(argv[argc][0])
		{
			// Optionen
			case '-':
				switch(argv[argc][1])
				{
					case 'p':
						CopyArgument(g_szPhonemFile, argv[argc] + 2, ".wav");
						break;
					case 'o':
						g_bKeepDeviceOpen = (argv[argc][2] != '0');
						break;
					default:
						printf("Unbekannte Option: %c\n", argv[argc][1]);
				}
				break;
			// Textdatei
			default:
				CopyArgument(g_szTextFile, argv[argc], ".txt");
		}
	}

	// Vorhandene Audio-Wiedergabegerte ermitteln
	InitDeviceList();

	// Phonemdatei (Audiodaten und Phonemdaten) laden
	if (!LoadWavFile())
		return 1;

	if (g_nCueCount == 0)
	{
		printf("Fehler: In der Phonemdatei \"%s\" sind keine Phoneme definiert.\n", g_szPhonemFile);
		return 1;
	}

	SortCueArray(g_pCueArray);

	// Wenn in der Kommandozeile eine Textdatei angegeben ist,
	// diese vorlesen und Programm beenden.
	if(*g_szTextFile)
		return PlayTextFile(false, false, false) ? 0 : 1; // 0 = OK, 1 = Fehler

//	Sprich("wilk0m3ntzumprogram|Spric.");
//	Sprich("wnsi:fra:g3nha:b3n  dange:b3nsi:d0x ");
//	Sprich("ainfaxdenb3fe:l  hilf3:  ain");

	PrintHelpText();
	printf("Es sind %d Phoneme definiert.\n\n", g_nCueCount);

	char szPreviousInput[256] = "";
	char szSentence[256] = "";

	// Endlosschleife zur Verarbeitung der Tastatureingaben
	while(1)
	{
		char szInput[256];

		// Eingabeprompt
		printf("Sprich>");
		GetString(szInput, sizeof(szInput));

		// Bei Leereingabe letzte Eingabe wiederverwenden
		if(!*szInput) StringCopy(szInput, szPreviousInput);

		// Eingaben auswerten
		if(!strncmp(szInput, CMD_LADE, SIZE_LADE))
		{
			CopyArgument(g_szTextFile, szInput + SIZE_LADE, ".txt");
			ResetTextfileParameters();
			printf("Geladene Textdatei: %s\n", g_szTextFile);
		}
		else if(!strncmp(szInput, CMD_LIES, SIZE_LIES))
		{
			StringCopy(szPreviousInput, szInput);
			CopyArgument(g_szTextFile, szInput + SIZE_LIES, ".txt");
			ResetTextfileParameters();
			PlayTextFile(false, false, false);
		}
		else if(!strncmp(szInput, CMD_SCHREIBE, SIZE_SCHREIBE))
		{
			CopyArgument(g_szTextFile, szInput + SIZE_SCHREIBE, ".txt");
			ResetTextfileParameters();
			AppendToTextFile(szSentence);
		}
		else if(!strncmp(szInput, CMD_WAVCUE, SIZE_WAVCUE)) // muss vor "cue" stehen!
		{
			CopyArgument(g_szWavFile, szInput + SIZE_WAVCUE, ".wav");
			SetFilenameIfEmpty(g_szWavFile, g_szTextFile, ".wav");
			PlayTextFile(true, true, false);
		}
		else if(!strncmp(szInput, CMD_WAVPLAY, SIZE_WAVPLAY)) // muss vor "cue" stehen!
		{
			CopyArgument(g_szWavFile, szInput + SIZE_WAVPLAY, ".wav");
			SetFilenameIfEmpty(g_szWavFile, g_szTextFile, ".wav");
			PlayTextFile(true, false, true);
		}
		else if(!strncmp(szInput, CMD_WAV, SIZE_WAV))
		{
			CopyArgument(g_szWavFile, szInput + SIZE_WAV, ".wav");
			SetFilenameIfEmpty(g_szWavFile, g_szTextFile, ".wav");
			PlayTextFile(true, false, false);
		}
		else if(!strncmp(szInput, CMD_ZEILE, SIZE_ZEILE))
		{
			CopyArgument(g_szLineRange, szInput + SIZE_ZEILE, "");
			ParseLineRange();
		}
		else if(!strcmp(szInput, CMD_LISTE))
		{
			PrintPhonemList();
		}
		else if(!strncmp(szInput, CMD_GERAET, SIZE_GERAET))
		{
			if(!*(szInput + SIZE_GERAET))
			{
				printf("Vorhandene Gerte:\n");
				ListDevices();
				printf("Auswahl: ");
				GetString(szInput + SIZE_GERAET, sizeof(szInput) - SIZE_GERAET);
			}
			if(*(szInput + SIZE_GERAET))
				sscanf(szInput + SIZE_GERAET, "%d", &g_nDeviceId);

			printf("Ausgewhltes Gert: %s\n", GetDeviceName(g_nDeviceId));
		}
		else if(!strcmp(szInput, CMD_HILFE))
		{
			PrintHelpText();
		}
		else if(!strcmp(szInput, CMD_ENDE))
		{
			break;
		}
		else
		{
			StringCopy(szPreviousInput, szInput);
			StringCopy(szSentence, szInput);
			Sprich(szInput);
		}
	}

//	Sprich("tSz");
	free(g_pWaveData);
	free(g_pCueArray);

	return 0;
}
