/*
	Script zur Berechnung von Sonne/Mond-Formeln
	Quelle: http://lexikon.astronomie.info/java/sunmoon/
	Stand: 03.03.2014

	Abgendert durch Stefan Bion in folgenden Punkten
	- Init-Funktion in Compute-Funktion integriert
	- Parameter Lnge und Breite fr Compute-Funktion
	- globale Variablen statt Formularfelder
	- alle brigen Variablen lokal statt global
	- Prefix "sm" (= "SonneMond") fr alle globalen Funktions- und Variablennamen (eigener Namespace)
	- Ausgabe der Uhrzeit ohne Dezimalwert (nur HH:MM[:SS])
	- Ausgabe der Stunde der Uhrzeit ohne fhrende 0
	- Zustzliche Ausgabe der Indexnummer der Mondphase, um damit z.B. Bilder referenzieren zu knnen

	Es folgt der Original-Kommentar zum Script vom Autor Arnold Barmettler:

	------------------ Das Script beginnt hier ---------------
	KOMMENTAR ZUM SCRIPT 
	Wir freuen uns selbstverstaendlich wenn Sie sich
	fuer die Details unseres kleinen Skripts interessieren. 
	Es bringt jedoch nichts, wenn Sie dieses Script auf Ihre Seite kopieren. 
	Ein einfacher Link beweist genau so gut, dass Sie das Skript gefunden haben.
	Kopieren Sie deshalb dieses Skript nicht auf Ihre 'private' Homepage.
	Arnold Barmettler, astro!nfo  
	 
	Entfernen Sie folgende Informationen auf keinen Fall: / Do not remove following text:
	Source code by Arnold Barmettler, www.astronomie.info / www.CalSky.com
	based on algorithms by Peter Duffett-Smith's great and easy book
	'Practical Astronomy with your Calculator'.
*/

var smPi = Math.PI;
var smDEG = smPi/180.0;
var smRAD = 180./smPi;


function smSqr(x)
{
	return x * x;
}


// return integer value, closer to 0
function smInt(x)
{
	return x < 0 ? Math.ceil(x) : Math.floor(x);
}

function smFrac(x)
{
	return x - Math.floor(x);
}

function smMod(a, b)
{
	return a - Math.floor(a / b) * b;
}


// Modulo PI
function smMod2Pi(x)
{
	return smMod(x, 2. * smPi);
}


function smRound100000(x) { return(Math.round(100000.*x)/100000.); }
function smRound10000(x) { return(Math.round(10000.*x)/10000.); }
function smRound1000(x) { return(Math.round(1000.*x)/1000.); }
function smRound100(x) { return(Math.round(100.*x)/100.); }
function smRound10(x) { return(Math.round(10.*x)/10.); }


function smHHMM(hh) 
{
	if (hh==0) return("--:--");
	
	var m = smFrac(hh)*60.;
	var h = smInt(hh);

	if (m>=59.5) { h++; m -=60.; }
	m = Math.round(m);
//	if (h<10) h = "0"+h;
	h = h+":";
	if (m<10) h = h+"0";
	h = h+m;

//	return(h+" = "+smRound1000(hh));
	return h;
}

function smHHMMSS(hh) 
{
	if (hh==0) return("--:--:--");
	
	var m = smFrac(hh)*60;
	var h = smInt(hh);
	var s = smFrac(m)*60.;

	m = smInt(m);
	if (s>=59.5) { m++; s -=60.; }
	if (m>=60)	 { h++; m -=60; }
	s = Math.round(s);
//	if (h<10) h = "0"+h;
	h = h+":";
	if (m<10) h = h+"0";
	h = h+m+":";
	if (s<10) h = h+"0";
	h = h+s;

//	return(h+" = "+smRound10000(hh));
	return h;
}


function smSign(lon)
{ 
	var signs= new Array("Widder", "Stier", "Zwillinge", "Krebs", "Lwe", "Jungfrau", 
		"Waage", "Skorpion", "Schtze", "Steinbock", "Wassermann", "Fische");
	return( signs[Math.floor(lon*smRAD/30)] );
}


// Calculate Julian date: valid only from 1.3.1901 to 28.2.2100
function smCalcJD(day,month,year)
{
	var jd = 2415020.5-64; // 1.1.1900 - correction of algorithm

	if (month<=2) { year--; month += 12; }
	jd += smInt( (year-1900)*365.25 );
	jd += smInt( 30.6001*(1+month) );

	return(jd + day);
}


// Julian Date to Greenwich Mean Sidereal Time
function smJD2GMST(JD)
{
	var UT = smFrac(JD-0.5)*24.; // UT in hours
	JD = Math.floor(JD-0.5)+0.5;	 // JD at 0 hours UT
	var T = (JD-2451545.0)/36525.0;
	var T0 = 6.697374558 + T*(2400.051336 + T*0.000025862);

	return(smMod(T0+UT*1.002737909, 24.));
}


// Convert Greenweek mean sidereal time to UT
function smGMST2UT(JD, gmst)
{
	JD = Math.floor(JD-0.5)+0.5;	 // JD at 0 hours UT
	var T = (JD-2451545.0)/36525.0;
	var T0 = smMod(6.697374558 + T*(2400.051336 + T*0.000025862), 24.);
	var UT = 0.9972695663*((gmst-T0));

	return(UT);
}


// Local Mean Sidereal Time, geographical longitude in radians, East is positive
function smGMST2LMST(gmst, lon)
{
	var lmst = smMod(gmst+smRAD*lon/15, 24.);

	return( lmst );
}


// Transform ecliptical coordinates (lon/lat) to equatorial coordinates (RA/dec)
function smEcl2Equ(coor, TDT)
{
	var T = (TDT-2451545.0)/36525.; // Epoch 2000 January 1.5
	var eps = (23.+(26+21.45/60.)/60. + T*(-46.815 +T*(-0.0006 + T*0.00181) )/3600. )*smDEG;
	var coseps = Math.cos(eps);
	var sineps = Math.sin(eps);
	
	var sinlon = Math.sin(coor.lon);
	coor.ra	= smMod2Pi( Math.atan2( (sinlon*coseps-Math.tan(coor.lat)*sineps), Math.cos(coor.lon) ) );
	coor.dec = Math.asin( Math.sin(coor.lat)*coseps + Math.cos(coor.lat)*sineps*sinlon );
	
	return coor;
}


// Transform equatorial coordinates (RA/Dec) to horizonal coordinates (azimuth/altitude)
// Refraction is ignored
function smEqu2Altaz(coor, TDT, geolat, lmst)
{
	var cosdec = Math.cos(coor.dec);
	var sindec = Math.sin(coor.dec);
	var lha = lmst - coor.ra;
	var coslha = Math.cos(lha);
	var sinlha = Math.sin(lha);
	var coslat = Math.cos(geolat);
	var sinlat = Math.sin(geolat);
	
	var N = -cosdec * sinlha;
	var D = sindec * coslat - cosdec * coslha * sinlat;
	coor.az = smMod2Pi( Math.atan2(N, D) );
	coor.alt = Math.asin( sindec * sinlat + cosdec * coslha * coslat );

	return coor;
}


// Transform geocentric equatorial coordinates (RA/Dec) to topocentric equatorial coordinates
function smGeoEqu2TopoEqu(coor, observer, lmst)
{
	var cosdec = Math.cos(coor.dec);
	var sindec = Math.sin(coor.dec);
	var coslst = Math.cos(lmst);
	var sinlst = Math.sin(lmst);
	var coslat = Math.cos(observer.lat); // we should use geocentric latitude, not geodetic latitude
	var sinlat = Math.sin(observer.lat);
	var rho = observer.radius; // observer-geocenter in Kilometer
	
	var x = coor.distance*cosdec*Math.cos(coor.ra) - rho*coslat*coslst;
	var y = coor.distance*cosdec*Math.sin(coor.ra) - rho*coslat*sinlst;
	var z = coor.distance*sindec - rho*sinlat;

	coor.distanceTopocentric = Math.sqrt(x*x + y*y + z*z);
	coor.decTopocentric = Math.asin(z/coor.distanceTopocentric);
	coor.raTopocentric = smMod2Pi( Math.atan2(y, x) );

	return coor;
}


// Calculate cartesian from polar coordinates
function smEquPolar2Cart( lon, lat, distance )
{
	var cart = new Object();
	var rcd = Math.cos(lat)*distance;

	cart.x = rcd*Math.cos(lon);
	cart.y = rcd*Math.sin(lon);
	cart.z = distance * Math.sin(lat);

	return(cart);
}


// Calculate observers cartesian equatorial coordinates (x,y,z in celestial frame) 
// from geodetic coordinates (longitude, latitude, height above WGS84 ellipsoid)
// Currently only used to calculate distance of a body from the observer
function smObserver2EquCart( lon, lat, height, gmst )
{
	var flat = 298.257223563;				// WGS84 flatening of earth
	var aearth = 6378.137;					 // GRS80/WGS84 semi major axis of earth ellipsoid
	var cart = new Object();
	// Calculate geocentric latitude from geodetic latitude
	var co = Math.cos (lat);
	var si = Math.sin (lat);
	var fl = 1.0 - 1.0 / flat;
	fl = fl * fl;
	si = si * si;
	var u = 1.0 / Math.sqrt (co * co + fl * si);
	var a = aearth * u + height;
	var b = aearth * fl * u + height;
	var radius = Math.sqrt (a * a * co * co + b * b * si); // geocentric distance from earth center
	cart.y = Math.acos (a * co / radius); // geocentric latitude, rad
	cart.x = lon; // longitude stays the same
	if (lat < 0.0) { cart.y = -cart.y; } // adjust sign
	cart = smEquPolar2Cart( cart.x, cart.y, radius ); // convert from geocentric polar to geocentric cartesian, with regard to Greenwich
	// rotate around earth's polar axis to align coordinate system from Greenwich to vernal equinox
	x=cart.x; y=cart.y;
	var rotangle = gmst/24*2*smPi; // sideral time gmst given in hours. Convert to radians
	cart.x = x*Math.cos(rotangle)-y*Math.sin(rotangle);
	cart.y = x*Math.sin(rotangle)+y*Math.cos(rotangle);
	cart.radius = radius;
	cart.lon = lon;
	cart.lat = lat;

	return(cart);
}


// Calculate coordinates for Sun
// Coordinates are accurate to about 10s (right ascension) 
// and a few minutes of arc (declination)
function smSunPosition(TDT, geolat, lmst)
{
	var D = TDT-2447891.5;
	
	var eg = 279.403303*smDEG;
	var wg = 282.768422*smDEG;
	var e	= 0.016713;
	var a	= 149598500; // km
	var diameter0 = 0.533128*smDEG; // angular diameter of Moon at a distance
	
	var MSun = 360*smDEG/365.242191*D+eg-wg;
	var nu = MSun + 360.*smDEG/smPi*e*Math.sin(MSun);
	
	var sunCoor = new Object();
	sunCoor.lon =	smMod2Pi(nu+wg);
	sunCoor.lat = 0;
	sunCoor.anomalyMean = MSun;
	
	sunCoor.distance = (1-smSqr(e))/(1+e*Math.cos(nu)); // distance in astronomical units
	sunCoor.diameter = diameter0/sunCoor.distance; // angular diameter in radians
	sunCoor.distance *= a;												 // distance in km
	sunCoor.parallax = 6378.137/sunCoor.distance;	// horizonal parallax

	sunCoor = smEcl2Equ(sunCoor, TDT);
	
	// Calculate horizonal coordinates of sun, if geographic positions is given
	if (geolat!=null && lmst!=null) {
		sunCoor = smEqu2Altaz(sunCoor, TDT, geolat, lmst);
	}
	
	sunCoor.sign = smSign(sunCoor.lon);

	return sunCoor;
}


// Calculate data and coordinates for the Moon
// Coordinates are accurate to about 1/5 degree (in ecliptic coordinates)
function smMoonPosition(sunCoor, TDT, observer, lmst)
{
	var D = TDT-2447891.5;
	
	// Mean Moon orbit elements as of 1990.0
	var l0 = 318.351648*smDEG;
	var P0 =	36.340410*smDEG;
	var N0 = 318.510107*smDEG;
	var i	= 5.145396*smDEG;
	var e	= 0.054900;
	var a	= 384401; // km
	var diameter0 = 0.5181*smDEG; // angular diameter of Moon at a distance
	var parallax0 = 0.9507*smDEG; // parallax at distance a
	
	var l = 13.1763966*smDEG*D+l0;
	var MMoon = l-0.1114041*smDEG*D-P0; // Moon's mean anomaly M
	var N = N0-0.0529539*smDEG*D;			 // Moon's mean ascending node longitude
	var C = l-sunCoor.lon;
	var Ev = 1.2739*smDEG*Math.sin(2*C-MMoon);
	var Ae = 0.1858*smDEG*Math.sin(sunCoor.anomalyMean);
	var A3 = 0.37*smDEG*Math.sin(sunCoor.anomalyMean);
	var MMoon2 = MMoon+Ev-Ae-A3;	// corrected Moon anomaly
	var Ec = 6.2886*smDEG*Math.sin(MMoon2);	// equation of centre
	var A4 = 0.214*smDEG*Math.sin(2*MMoon2);
	var l2 = l+Ev+Ec-Ae+A4; // corrected Moon's longitude
	var V = 0.6583*smDEG*Math.sin(2*(l2-sunCoor.lon));
	var l3 = l2+V; // true orbital longitude;

	var N2 = N-0.16*smDEG*Math.sin(sunCoor.anomalyMean);
	
	var moonCoor = new Object();	
	moonCoor.lon = smMod2Pi( N2 + Math.atan2( Math.sin(l3-N2)*Math.cos(i), Math.cos(l3-N2) ) );
	moonCoor.lat = Math.asin( Math.sin(l3-N2)*Math.sin(i) );
	moonCoor.orbitLon = l3;
	
	moonCoor = smEcl2Equ(moonCoor, TDT);
	// relative distance to semi mayor axis of lunar oribt
	moonCoor.distance = (1-smSqr(e)) / (1+e*Math.cos(MMoon2+Ec) );
	moonCoor.diameter = diameter0/moonCoor.distance; // angular diameter in radians
	moonCoor.parallax = parallax0/moonCoor.distance; // horizontal parallax in radians
	moonCoor.distance *= a; // distance in km

	// Calculate horizonal coordinates of sun, if geographic positions is given
	if (observer!=null && lmst!=null) {
		// transform geocentric coordinates into topocentric (==observer based) coordinates
	moonCoor = smGeoEqu2TopoEqu(moonCoor, observer, lmst);
	moonCoor.raGeocentric = moonCoor.ra; // backup geocentric coordinates
	moonCoor.decGeocentric = moonCoor.dec;
	moonCoor.ra=moonCoor.raTopocentric;
	moonCoor.dec=moonCoor.decTopocentric;
		moonCoor = smEqu2Altaz(moonCoor, TDT, observer.lat, lmst); // now ra and dec are topocentric
	}
	
	// Age of Moon in radians since New Moon (0) - Full Moon (smPi)
	moonCoor.moonAge = smMod2Pi(l3-sunCoor.lon);	 
	moonCoor.phase	 = 0.5*(1-Math.cos(moonCoor.moonAge)); // Moon phase, 0-1
	
	var phases = new Array("Neumond", "Zunehmende Sichel", "Erstes Viertel", "Zunehmender Mond", 
	 "Vollmond", "Abnehmender Mond", "Letztes Viertel", "Abnehmende Sichel", "Neumond");
	var mainPhase = 1./29.53*360*smDEG; // show 'Newmoon, 'Quarter' for +/-1 day arond the actual event
	var p = smMod(moonCoor.moonAge, 90.*smDEG);
	if (p < mainPhase || p > 90*smDEG-mainPhase) p = 2*Math.round(moonCoor.moonAge / (90.*smDEG));
	else p = 2*Math.floor(moonCoor.moonAge / (90.*smDEG))+1;
	moonCoor.moonPhase = phases[p];
	moonCoor.moonPhaseIndex = p;
	
	moonCoor.sign = smSign(moonCoor.lon);

	return(moonCoor);
}


// Rough refraction formula using standard atmosphere: 1015 mbar and 10C
// Input true altitude in radians, Output: increase in altitude in degrees
function smRefraction(alt)
{
	var altdeg = alt*smRAD;
	if (altdeg<-2 || altdeg>=90) return(0);
	 
	var pressure		= 1015;
	var temperature = 10;
	if (altdeg>15) return( 0.00452*pressure/( (273+temperature)*Math.tan(alt)) );
	
	var y = alt;
	var D = 0.0;
	var P = (pressure-80.)/930.;
	var Q = 0.0048*(temperature-10.);
	var y0 = y;
	var D0 = D;

	for (i=0; i<3; i++)
	{
		var N = y+(7.31/(y+4.4));
		N = 1./Math.tan(N*smDEG);
		D = N*P/(60.+Q*(N+39.));
		N = y-y0;
		y0 = D-D0-N;
		if ((N != 0.) && (y0 != 0.)) { N = y-N*(alt+D-y)/y0; }
		else { N = alt+D; }
		y0 = y;
		D0 = D;
		y = N;
	}
	return( D ); // Hebung durch Refraktion in radians
}


// returns Greenwich sidereal time (hours) of time of rise 
// and set of object with coordinates coor.ra/coor.dec
// at geographic position lon/lat (all values in radians)
// Correction for refraction and semi-diameter/parallax of body is taken care of in function RiseSet
// h is used to calculate the twilights. It gives the required elevation of the disk center of the sun
function smGMSTRiseSet(coor, lon, lat, h)
{
	var h = (h == null) ? 0. : h; // set default value
	var riseset = new Object();
//	var tagbogen = Math.acos(-Math.tan(lat)*Math.tan(coor.dec)); // simple formula if twilight is not required
	var tagbogen = Math.acos((Math.sin(h) - Math.sin(lat)*Math.sin(coor.dec)) / (Math.cos(lat)*Math.cos(coor.dec)));

	riseset.transit =		 smRAD/15*(				 +coor.ra-lon);
	riseset.rise		= 24.+smRAD/15*(-tagbogen+coor.ra-lon); // calculate GMST of rise of object
	riseset.set		 =		 smRAD/15*(+tagbogen+coor.ra-lon); // calculate GMST of set of object

	// using the modulo function Mod, the day number goes missing. This may get a problem for the moon
	riseset.transit = smMod(riseset.transit, 24);
	riseset.rise		= smMod(riseset.rise, 24);
	riseset.set		 = smMod(riseset.set, 24);

	return(riseset);
}


// Find GMST of rise/set of object from the two calculates 
// (start)points (day 1 and 2) and at midnight UT(0)
function smInterpolateGMST(gmst0, gmst1, gmst2, timefactor)
{
	return( (timefactor*24.07*gmst1- gmst0*(gmst2-gmst1)) / (timefactor*24.07+gmst1-gmst2) );
}


// JD is the Julian Date of 0h UTC time (midnight)
function smRiseSet(jd0UT, coor1, coor2, lon, lat, timeinterval, altitude)
{
	// altitude of sun center: semi-diameter, horizontal parallax and (standard) refraction of 34'
	var alt = 0.; // calculate 
	var altitude = (altitude == null) ? 0. : altitude; // set default value

	// true height of sun center for sunrise and set calculation. Is kept 0 for twilight (ie. altitude given):
	if (!altitude) alt = 0.5*coor1.diameter-coor1.parallax+34./60*smDEG; 
	
	var rise1 = smGMSTRiseSet(coor1, lon, lat, altitude);
	var rise2 = smGMSTRiseSet(coor2, lon, lat, altitude);
	
	var rise = new Object();
	
	// unwrap GMST in case we move across 24h -> 0h
	if (rise1.transit > rise2.transit && Math.abs(rise1.transit-rise2.transit)>18) rise2.transit += 24;
	if (rise1.rise		> rise2.rise		&& Math.abs(rise1.rise	 -rise2.rise)>18)		rise2.rise += 24;
	if (rise1.set		 > rise2.set		 && Math.abs(rise1.set		-rise2.set)>18)		 rise2.set	+= 24;
	var T0 = smJD2GMST(jd0UT);
	//	var T02 = T0-zone*1.002738; // Greenwich sidereal time at 0h time zone (zone: hours)

	// Greenwich sidereal time for 0h at selected longitude
	var T02 = T0-lon*smRAD/15*1.002738; if (T02 < 0) T02 += 24; 

	if (rise1.transit < T02) { rise1.transit += 24; rise2.transit += 24; }
	if (rise1.rise		< T02) { rise1.rise		+= 24; rise2.rise		+= 24; }
	if (rise1.set		 < T02) { rise1.set		 += 24; rise2.set		 += 24; }
	
	// Refraction and Parallax correction
	var decMean = 0.5*(coor1.dec+coor2.dec);
	var psi = Math.acos(Math.sin(lat)/Math.cos(decMean));
	var y = Math.asin(Math.sin(alt)/Math.sin(psi));
	var dt = 240*smRAD*y/Math.cos(decMean)/3600; // time correction due to refraction, parallax

	rise.transit = smGMST2UT( jd0UT, smInterpolateGMST( T0, rise1.transit, rise2.transit, timeinterval) );
	rise.rise		= smGMST2UT( jd0UT, smInterpolateGMST( T0, rise1.rise,		rise2.rise,		timeinterval) -dt );
	rise.set		 = smGMST2UT( jd0UT, smInterpolateGMST( T0, rise1.set,		 rise2.set,		 timeinterval) +dt );
	
	return(rise);	
}


// Find (local) time of sunrise and sunset, and twilights
// JD is the Julian Date of 0h local time (midnight)
// Accurate to about 1-2 minutes
// recursive: 1 - calculate rise/set in UTC in a second run
// recursive: 0 - find rise/set on the current local day. This is set when doing the first call to this function
function smCalcSunRise(JD, deltaT, lon, lat, zone, recursive)
{
	var jd0UT = Math.floor(JD-0.5)+0.5;	 // JD at 0 hours UT
	var coor1 = smSunPosition(jd0UT+	deltaT/24./3600.);
	var coor2 = smSunPosition(jd0UT+1.+deltaT/24./3600.); // calculations for next day's UTC midnight
	
	var risetemp = new Object();
	var rise = new Object();
	// rise/set time in UTC. 
	rise = smRiseSet(jd0UT, coor1, coor2, lon, lat, 1); 
	if (!recursive) { // check and adjust to have rise/set time on local calendar day
		if (zone>0) {
			// rise time was yesterday local time -> calculate rise time for next UTC day
			if (rise.rise>=24-zone || rise.transit>=24-zone || rise.set>=24-zone) {
				risetemp = smCalcSunRise(JD+1, deltaT, lon, lat, zone, 1);
				if (rise.rise>=24-zone) rise.rise = risetemp.rise;
				if (rise.transit >=24-zone) rise.transit = risetemp.transit;
				if (rise.set >=24-zone) rise.set	= risetemp.set;
			}
		}
		else if (zone<0) {
			// rise time was yesterday local time -> calculate rise time for next UTC day
			if (rise.rise<-zone || rise.transit<-zone || rise.set<-zone) {
				risetemp = smCalcSunRise(JD-1, deltaT, lon, lat, zone, 1);
				if (rise.rise<-zone) rise.rise = risetemp.rise;
				if (rise.transit<-zone) rise.transit = risetemp.transit;
				if (rise.set <-zone) rise.set	= risetemp.set;
			}
		}
	
		rise.transit = smMod(rise.transit+zone, 24.);
		rise.rise		= smMod(rise.rise	 +zone, 24.);
		rise.set		 = smMod(rise.set		+zone, 24.);

	// Twilight calculation
	// civil twilight time in UTC. 
	risetemp = smRiseSet(jd0UT, coor1, coor2, lon, lat, 1, -6.*smDEG);
	rise.cicilTwilightMorning = smMod(risetemp.rise +zone, 24.);
	rise.cicilTwilightEvening = smMod(risetemp.set	+zone, 24.);

	// nautical twilight time in UTC. 
	risetemp = smRiseSet(jd0UT, coor1, coor2, lon, lat, 1, -12.*smDEG);
	rise.nauticalTwilightMorning = smMod(risetemp.rise +zone, 24.);
	rise.nauticalTwilightEvening = smMod(risetemp.set	+zone, 24.);

	// astronomical twilight time in UTC. 
	risetemp = smRiseSet(jd0UT, coor1, coor2, lon, lat, 1, -18.*smDEG);
	rise.astronomicalTwilightMorning = smMod(risetemp.rise +zone, 24.);
	rise.astronomicalTwilightEvening = smMod(risetemp.set	+zone, 24.);
	}
	return( rise );	
}



// Find local time of moonrise and moonset
// JD is the Julian Date of 0h local time (midnight)
// Accurate to about 5 minutes or better
// recursive: 1 - calculate rise/set in UTC
// recursive: 0 - find rise/set on the current local day (set could also be first)
// returns '' for moonrise/set does not occur on selected day
function smCalcMoonRise(JD, deltaT, lon, lat, zone, recursive)
{
	var timeinterval = 0.5;
	
	var jd0UT = Math.floor(JD-0.5)+0.5;	 // JD at 0 hours UT
	var suncoor1 = smSunPosition(jd0UT+ deltaT/24./3600.);
	var coor1 = smMoonPosition(suncoor1, jd0UT+ deltaT/24./3600.);

	var suncoor2 = smSunPosition(jd0UT +timeinterval + deltaT/24./3600.); // calculations for noon
	// calculations for next day's midnight
	var coor2 = smMoonPosition(suncoor2, jd0UT +timeinterval + deltaT/24./3600.); 
	
	var risetemp = new Object();
	var rise = new Object();
	
	// rise/set time in UTC, time zone corrected later.
	// Taking into account refraction, semi-diameter and parallax
	rise = smRiseSet(jd0UT, coor1, coor2, lon, lat, timeinterval); 
	
	if (!recursive) { // check and adjust to have rise/set time on local calendar day
		if (zone>0) {
			// recursive call to MoonRise returns events in UTC
			riseprev = smCalcMoonRise(JD-1., deltaT, lon, lat, zone, 1); 
			
			if (rise.transit >= 24.-zone || rise.transit < -zone) { // transit time is tomorrow local time
				if (riseprev.transit < 24.-zone) rise.transit = ''; // there is no moontransit today
				else rise.transit	= riseprev.transit;
			}

			if (rise.rise >= 24.-zone || rise.rise < -zone) { // transit time is tomorrow local time
				if (riseprev.rise < 24.-zone) rise.rise = ''; // there is no moontransit today
				else rise.rise	= riseprev.rise;
			}

			if (rise.set >= 24.-zone || rise.set < -zone) { // transit time is tomorrow local time
				if (riseprev.set < 24.-zone) rise.set = ''; // there is no moontransit today
				else rise.set	= riseprev.set;
			}

		}
		else if (zone<0) {
			// rise/set time was tomorrow local time -> calculate rise time for former UTC day
			if (rise.rise<-zone || rise.set<-zone || rise.transit<-zone) { 
				risetemp = smCalcMoonRise(JD+1., deltaT, lon, lat, zone, 1);
				
				if (rise.rise < -zone) {
					if (risetemp.rise > -zone) rise.rise = ''; // there is no moonrise today
					else rise.rise = risetemp.rise;
				}
				
				if (rise.transit < -zone)
				{
					if (risetemp.transit > -zone)	rise.transit = ''; // there is no moonset today
					else rise.transit	= risetemp.transit;
				}
				
				if (rise.set < -zone)
				{
					if (risetemp.set > -zone)	rise.set = ''; // there is no moonset today
					else rise.set	= risetemp.set;
				}
				
			}
		}
		
		if (rise.rise)		rise.rise = smMod(rise.rise+zone, 24.);		// correct for time zone, if time is valid
		if (rise.transit) rise.transit	= smMod(rise.transit +zone, 24.); // correct for time zone, if time is valid
		if (rise.set)		 rise.set	= smMod(rise.set +zone, 24.);		// correct for time zone, if time is valid
	}
	return( rise );	
}



function smCompute(smLon, smLat)
{
	  //////////
	 // Init //
	//////////

	var now=new Date();
	var smHour		= now.getHours();
	var smMinute	= now.getMinutes();
	var smSecond	= now.getSeconds();
	var smDay		= now.getDate();
	var smMonth		= now.getMonth()+1;
	var smYear		= now.getYear();
	if(now.getYear() < 1900) smYear += 1900; // MSIE returns 2004, but NS years since 1900
	var smZone		= -now.getTimezoneOffset() / 60.;
	var smDeltaT	= 65; // deltaT - difference among 'earth center' versus 'observered' time (TDT-UT), in seconds

	  /////////////
	 // Compute //
	/////////////

	if(smYear <= 1900 || smYear >= 2100)
	{
	//	alert("Dies Script erlaubt nur Berechnungen in der Zeitperiode 1901-2099. Angezeigte Resultat sind ungltig.");
		return;
	}

	var JD0 = smCalcJD(smDay, smMonth, smYear);
	var JD = JD0 + (smHour - smZone + smMinute / 60. + smSecond / 3600.) / 24.;
	var TDT = JD + smDeltaT / 24. / 3600.;

	var lat = smLat * smDEG; // geodetic latitude of observer on WGS84
	var lon = smLon * smDEG; // latitude of observer
	var height = 0 * 0.001; // altiude of observer in meters above WGS84 ellipsoid (and converted to kilometers)

	var gmst = smJD2GMST(JD);
	var lmst = smGMST2LMST(gmst, lon);
	
	var observerCart = smObserver2EquCart(lon, lat, height, gmst); // geocentric cartesian coordinates of observer
 
	var sunCoor = smSunPosition(TDT, lat, lmst * 15. * smDEG); // Calculate data for the Sun at given time
	var moonCoor = smMoonPosition(sunCoor, TDT, observerCart, lmst * 15. * smDEG); // Calculate data for the Moon at given time

	smJD = smRound100000(JD);
	smGMST = smHHMMSS(gmst);
	smLMST = smHHMMSS(lmst);

	smSunLon	= smRound1000(sunCoor.lon * smRAD);
	smSunRA	 = smHHMM(sunCoor.ra * smRAD / 15);
	smSunDec	= smRound1000(sunCoor.dec * smRAD);
	smSunAz	 = smRound100(sunCoor.az * smRAD);
	smSunAlt	= smRound10(sunCoor.alt * smRAD + smRefraction(sunCoor.alt)); // including refraction

	smSunSign = sunCoor.sign;
	smSunDiameter = smRound100(sunCoor.diameter * smRAD * 60.); // angular diameter in arc seconds
	smSunDistance = smRound10(sunCoor.distance);

	// Calculate distance from the observer (on the surface of earth) to the center of the sun
	var sunCart = smEquPolar2Cart(sunCoor.ra, sunCoor.dec, sunCoor.distance);
	smSunDistanceObserver = smRound10(Math.sqrt(smSqr(sunCart.x - observerCart.x) + smSqr(sunCart.y - observerCart.y) + smSqr(sunCart.z - observerCart.z)));

	// JD0: JD of 0h UTC time
	var sunRise = smCalcSunRise(JD0, smDeltaT, lon, lat, smZone, 0);

	smSunTransit = smHHMM(sunRise.transit);
	smSunRise = smHHMM(sunRise.rise);
	smSunSet = smHHMM(sunRise.set);

	smSunCivilTwilightMorning = smHHMM(sunRise.cicilTwilightMorning);
	smSunCivilTwilightEvening = smHHMM(sunRise.cicilTwilightEvening);
	smSunNauticalTwilightMorning = smHHMM(sunRise.nauticalTwilightMorning);
	smSunNauticalTwilightEvening = smHHMM(sunRise.nauticalTwilightEvening);
	smSunAstronomicalTwilightMorning = smHHMM(sunRise.astronomicalTwilightMorning);
	smSunAstronomicalTwilightEvening = smHHMM(sunRise.astronomicalTwilightEvening);

	smMoonLon = smRound1000(moonCoor.lon * smRAD);
	smMoonLat = smRound1000(moonCoor.lat * smRAD);
	smMoonRA = smHHMM(moonCoor.ra * smRAD / 15.);
	smMoonDec = smRound1000(moonCoor.dec * smRAD);
	smMoonAz = smRound100(moonCoor.az * smRAD);
	smMoonAlt = smRound10(moonCoor.alt * smRAD + smRefraction(moonCoor.alt)); // including refraction
	smMoonAge = smRound1000(moonCoor.moonAge * smRAD);
	smMoonPhaseNumber = smRound1000(moonCoor.phase);
	smMoonPhase = moonCoor.moonPhase;
	smMoonPhaseIndex = moonCoor.moonPhaseIndex;

	smMoonSign = moonCoor.sign;
	smMoonDistance = smRound10(moonCoor.distance);
	smMoonDiameter = smRound100(moonCoor.diameter * smRAD * 60.); // angular diameter in arc seconds

	// Calculate distance from the observer (on the surface of earth) to the center of the moon
	var moonCart = smEquPolar2Cart(moonCoor.raGeocentric, moonCoor.decGeocentric, moonCoor.distance);
	smMoonDistanceObserver = smRound10(Math.sqrt(smSqr(moonCart.x - observerCart.x) + smSqr(moonCart.y - observerCart.y) + smSqr(moonCart.z - observerCart.z)));

	var moonRise = smCalcMoonRise(JD0, smDeltaT, lon, lat, smZone, 0);
	var moonRiseYesterday = smCalcMoonRise(JD0 - 1, smDeltaT, lon, lat, smZone, 0);
	var moonRiseTomorrow = smCalcMoonRise(JD0 + 1, smDeltaT, lon, lat, smZone, 0);

	smMoonTransit = smHHMM(moonRise.transit);
	smMoonRise = smHHMM(moonRise.rise != '' ? moonRise.rise : moonRiseYesterday.rise);
	smMoonSet = smHHMM(moonRise.set != '' ? moonRise.set : moonRiseTomorrow.set);
}

