#!/usr/bin/perl

################################################################################
#
# AutoGallery - Erstellt automatisch eine Bilder-Galerie.
#
# (c) 2011 by Stefan Bion
# http://www.StefanBion.de
#
################################################################################

use strict;

use CGI::Carp qw(fatalsToBrowser);

use Time::Local;
use Fcntl qw(:DEFAULT :flock);
use Digest::MD5 qw(md5_hex);
use File::Copy;

#################
# Konfiguration #
#################

my $gScriptName = $ENV{'SCRIPT_NAME'};
my $gDocumentRoot = $ENV{'DOCUMENT_ROOT'};

my $gImageDir = "/Beispielgalerie";
my $gImageRoot = "$gDocumentRoot$gImageDir";

my $gFileNameTemplateFramesetHtml = "$gDocumentRoot/Beispielgalerie/gallery.htm";
my $gFileNameTemplateFrameMenuHtml = "$gDocumentRoot/Beispielgalerie/gallery_menu.htm";
my $gFileNameTemplateFrameContentHtml = "$gDocumentRoot/Beispielgalerie/gallery_content.htm";
my $gFileNameTemplateFrameVotedHtml = "$gDocumentRoot/Beispielgalerie/gallery_voted.htm";
my $gFileNameImageVotes = "$gImageRoot/voted_images.db";	# Datei fr Bildbewertungen (optional)
my $gFileNameImageVotesBackupFormat = "$gImageRoot/voted_images_%s.db";	# Format-String des Dateinames der Backup-Datei fr Bildbewertungen (optional)

my $gImagesPerPage = 100;

# Parameter fr Gruppierung der Bilder
my $gstrGroupPatternDefault = "yymmdd-hhnnss";	# y = Jahr, m = Monat, d = Tag, h = Stunde, n = Minute, s = Sekunde
my $gnGroupSecondsDefault = 61;	# Differenzwert in Sekunden, ab dem gruppiert wird

# Benutzer und Pawrter der Administratoren
# Das Pawort wird als MD5-Hash in hexadezimaler Form angegeben und kann mit md5.pl erzeugt werden.
my %gUsers =
(
	'demo' => 'fe01ce2a7fbac8fafaed7c982a04e229'
);

my $gCookiePrefix = "BeispielAutoGallery";	# Eindeutiger Cookie-Namensprfix fr dieses CGI-Script auf diesem Server

###################
# Initialisierung #
###################

my $gbLoggedIn = 0;	# Kenner, ob Benutzer eingeloggt (1) oder nicht (0)
my $gUserName = '';	# Name des eingeloggten Benutzers

# Formulardaten parsen
my %FORM;
my %gSelectedImages;
ParseFormData(\%FORM, \%gSelectedImages);

# Cookies parsen
my %gCookies;
ParseCookies(\%gCookies);

# Hash fr bewertete Bilder
my %gVotedImages;
my %gNewVotedImages;
my $gbVotedImagesUpdated = 0;
my $gbVotedImagesReadError = 0;

my $gbGroup = '' ne $FORM{'group'} ? 1 : 0;	# Gruppierung ja oder nein
my $gstrGroupPattern = '' ne $FORM{'pattern'} ? $FORM{'pattern'} : $gstrGroupPatternDefault;
my $gnGroupSeconds = '' ne $FORM{'distance'} ? $FORM{'distance'} : $gnGroupSecondsDefault;
my $gnGroupLastSeconds;	# Merkvariable fr letzten Wert

#############
# Verteiler #
#############

# HTTP-Header incl. Cookies ausgeben
PrintHttpHeader();

#DEBUG:
#print "<p>FORM: ";
#for my $name (keys %FORM) { printf("%s = \"%s\", ", $name, $FORM{$name}); }

if($FORM{'print'} eq 'menu')
{
	# Men-Frame ausgeben
	PrintMenu();
}
elsif($FORM{'print'} eq 'album')
{
	if($FORM{'album'} eq 'voted')
	{
		# Bewertete Bilder ausgeben
		PrintVoted();
	}
	else
	{
		my $a_mode;

		if($FORM{'vote'})
		{
			$a_mode = 'vote';
		}
		elsif($FORM{'delete'} && $gbLoggedIn)
		{
			$a_mode = 'del';
		}

		# Album-Frame ausgeben
		PrintAlbum($a_mode);
	}
}
else
{
	# Frameset ausgeben
	PrintFrameset();
}

exit 0;

##############
# Funktionen #
##############

sub PrintFrameset
{
	# HTML-Template einlesen:
	open(TEMPLATE, "$gFileNameTemplateFramesetHtml") or die "File not found: $gFileNameTemplateFramesetHtml";
	my @TemplateZeilen = <TEMPLATE>;
	close(TEMPLATE);

	# Variablen ersetzen
	@TemplateZeilen = ReplaceLines(\@TemplateZeilen, '@@MENU_URL@@', "$gScriptName?print=menu");

	# Die HTML-Daten ausgeben
	print @TemplateZeilen;
}

sub PrintMenu
{
	# HTML-Template einlesen:
	open(TEMPLATE_MENU, "$gFileNameTemplateFrameMenuHtml") or die "File not found: $gFileNameTemplateFrameMenuHtml";
	my @TemplateZeilen = <TEMPLATE_MENU>;
	close(TEMPLATE_MENU);

	my $bSortOrderReverse = 'reverse' eq $FORM{'order'} ? 1 : 0;

	my $MenuZeilen;
	my $TemplateAlbum = ExtractAndReplaceLines(\@TemplateZeilen, '<album>', '</album>', '@@MENU@@');

	if(opendir(HANDLE_IMAGE_ROOT, "$gImageRoot"))
	{
		for my $strAlbumName (sort { $bSortOrderReverse ? ($b cmp $a) : ($a cmp $b) } readdir(HANDLE_IMAGE_ROOT))
		{
			next if(0 == index($strAlbumName, '.'));

			my $strAlbumUrl = "$gScriptName?print=album&album=$strAlbumName";

			my $nAlbumCount = 0;

			if(opendir(HANDLE_ALBUM_ROOT, "$gImageRoot/$strAlbumName"))
			{
				for my $strImageFileName (readdir(HANDLE_ALBUM_ROOT))
				{
					$nAlbumCount++ if($strImageFileName =~ /.jpg$|.gif$|.png$/)
				}
			}

			if($nAlbumCount > 0)
			{
				# Variablen ersetzen
				my $currentTemplateAlbum = $TemplateAlbum;
				$currentTemplateAlbum = ReplaceLine($currentTemplateAlbum, '@@ALBUM_URL@@', "$strAlbumUrl");
				$currentTemplateAlbum = ReplaceLine($currentTemplateAlbum, '@@ALBUM_NAME@@', "$strAlbumName");
				$currentTemplateAlbum = ReplaceLine($currentTemplateAlbum, '@@ALBUM_COUNT@@', "$nAlbumCount");

				$MenuZeilen .= $currentTemplateAlbum;
			}
		}
	}

	# Variablen ersetzen
	@TemplateZeilen = ReplaceLines(\@TemplateZeilen, '@@MENU@@', "$MenuZeilen");
	@TemplateZeilen = ReplaceLines(\@TemplateZeilen, '@@SCRIPT_URL@@', "$gScriptName");
	@TemplateZeilen = ReplaceLines(\@TemplateZeilen, '@@VOTED_COUNT@@', ReadImageVotes());

	# Die HTML-Daten ausgeben
	print @TemplateZeilen;
}

sub PrintVoted
{
	# HTML-Template einlesen:
	open(TEMPLATE_VOTED, "$gFileNameTemplateFrameVotedHtml") or die "File not found: $gFileNameTemplateFrameVotedHtml";
	my @TemplateZeilen = <TEMPLATE_VOTED>;
	close(TEMPLATE_VOTED);

	my $nPage = '' ne $FORM{'page'} ? $FORM{'page'} : 1;

	$nPage -= 1;
	$nPage = 0 if($nPage < 0);

	my $strImageZeilen;
	my $strTemplateImage = ExtractAndReplaceLines(\@TemplateZeilen, '<image>', '</image>', '@@IMAGE@@');

	my $nImageCount = 0;

	if('' ne "$gFileNameImageVotes" && open(IMAGE_VOTES, "$gFileNameImageVotes"))
	{
#		if(flock(IMAGE_VOTES, LOCK_EX))
		{
			my @Zeilen = <IMAGE_VOTES>;
			close(IMAGE_VOTES);

			# Eintrge sortieren
			@Zeilen = sort { SortVotedImages($a, $b) } @Zeilen;

			for my $Zeile (@Zeilen)
			{
				$nImageCount++;

				next if($nImageCount < $nPage * $gImagesPerPage + 1);
				next if($nImageCount > $nPage * $gImagesPerPage + $gImagesPerPage);

				chomp $Zeile;
				my ($nVotes, $strImageRelPath) = split(/\|/, $Zeile);

				# Dateiname aus Pfad extrahieren
				my $strImageFileName = $strImageRelPath;
				my @arrPathParts = split(/\//, $strImageRelPath);
				$strImageFileName = $arrPathParts[-1] if(@arrPathParts > 0);

				# Variablen ersetzen
				my $strCurrentTemplateImage = $strTemplateImage;
				$strCurrentTemplateImage = ReplaceLine($strCurrentTemplateImage, '@@IMAGE_NUMBER@@', $nImageCount);
				$strCurrentTemplateImage = ReplaceLine($strCurrentTemplateImage, '@@IMAGE_URL@@', "$gImageDir/$strImageRelPath");
				$strCurrentTemplateImage = ReplaceLine($strCurrentTemplateImage, '@@IMAGE_FILENAME@@', "$strImageFileName");
				$strCurrentTemplateImage = ReplaceLine($strCurrentTemplateImage, '@@IMAGE_VOTES@@', $nVotes);

				$strImageZeilen .= $strCurrentTemplateImage;
			}
		}
	}

	# Formular-Variablen fr Links zum Blttern

	my $urlParameters;
	$urlParameters .= "&print=$FORM{'print'}" if('' ne $FORM{'print'});
	$urlParameters .= "&album=$FORM{'album'}" if('' ne $FORM{'album'});

	# Links zum Blttern

	my $strPageLinks;

	my $nNumPages = int(($nImageCount + $gImagesPerPage - 1) / $gImagesPerPage);

	if($nNumPages > 1)
	{
		# "Zurck"
		if($nPage == 0)
		{
			$strPageLinks .= "[&lt;&lt;] ";
		}
		else
		{
			$strPageLinks .= "[<a href=\"$gScriptName?$urlParameters&page=" . $nPage . "\">&lt;&lt;</a>] ";
		}

		# Seitenzahlen
		for(my $nCurrentPage = 0; $nCurrentPage < $nNumPages; $nCurrentPage++)
		{
			if($nCurrentPage == $nPage)
			{
				$strPageLinks .= "[" . ($nCurrentPage + 1) . "] ";
			}
			else
			{
				$strPageLinks .= "[<a href=\"$gScriptName?$urlParameters&page=" . ($nCurrentPage + 1) . "\">" . ($nCurrentPage + 1) . "</a>] ";
			}
		}

		# "Weiter"
		if($nPage >= $nNumPages - 1)
		{
			$strPageLinks .= "[&gt;&gt;] ";
		}
		else
		{
			$strPageLinks .= "[<a href=\"$gScriptName?$urlParameters&page=" . ($nPage + 2) . "\">&gt;&gt;</a>] ";
		}
	}

	# Variablen ersetzen
	@TemplateZeilen = ReplaceLines(\@TemplateZeilen, '@@IMAGE@@', "$strImageZeilen");
	@TemplateZeilen = ReplaceLines(\@TemplateZeilen, '@@IMAGE_COUNT@@', "$nImageCount");
	@TemplateZeilen = ReplaceLines(\@TemplateZeilen, '@@PAGE@@', $nPage + 1);
	@TemplateZeilen = ReplaceLines(\@TemplateZeilen, '@@PAGES@@', "$nNumPages");
	@TemplateZeilen = ReplaceLines(\@TemplateZeilen, '@@PAGE_LINKS@@', "$strPageLinks");
	@TemplateZeilen = ReplaceLines(\@TemplateZeilen, '@@BR_PAGE_LINKS@@', '' ne $strPageLinks ? "<br>$strPageLinks" : '');

	# Die HTML-Daten ausgeben
	print @TemplateZeilen;
}

sub PrintAlbum
{
	my $a_mode = shift;

	# Bildbewertungen lesen
	ReadImageVotes();

	# HTML-Template einlesen:
	open(TEMPLATE_ALBUM, "$gFileNameTemplateFrameContentHtml") or die "File not found: $gFileNameTemplateFrameContentHtml";
	my @TemplateZeilen = <TEMPLATE_ALBUM>;
	close(TEMPLATE_ALBUM);

	my $strAlbumName = $FORM{'album'};
	my $bSortByDate = 'date' eq $FORM{'sort'} ? 1 : 0;
	my $bSortOrderReverse = 'reverse' eq $FORM{'order'} ? 1 : 0;
	my $nPage = '' ne $FORM{'page'} ? $FORM{'page'} : 1;

	$nPage -= 1;
	$nPage = 0 if($nPage < 0);

	my $strImageZeilen;
	my $strTemplateImage = ExtractAndReplaceLines(\@TemplateZeilen, '<image>', '</image>', '@@IMAGE@@');
	my $strTemplateGroup = ExtractAndReplaceLines(\@TemplateZeilen, '<groupheader>', '</groupheader>', '');
	my $strTemplateNoGroup = ExtractAndReplaceLines(\@TemplateZeilen, '<nogroupheader>', '</nogroupheader>', $gbGroup ? '' : '@@NOGROUP@@');
	my $strTemplateGroupPageHeader = ExtractAndReplaceLines(\@TemplateZeilen, '<grouppageheader>', '</grouppageheader>', $gbGroup ? '@@GROUP_PAGEHEADER@@' : '');
	my $strTemplateAdminControls = ExtractAndReplaceLines(\@TemplateZeilen, '<admincontrols>', '</admincontrols>', '');
	my $strTemplateImageCheckbox = ExtractAndReplaceLines(\@TemplateZeilen, '<imagecheckbox>', '</imagecheckbox>', '');
	my $strLogoutForm = ExtractAndReplaceLines(\@TemplateZeilen, '<loggedin>', '</loggedin>', '@@LOGOUT_FORM@@');
	my $strLoginForm = ExtractAndReplaceLines(\@TemplateZeilen, '<loggedout>', '</loggedout>', '@@LOGIN_FORM@@');

	# Template-Elemente ohne Zeilenumbruch:
	chomp $strTemplateAdminControls;
	chomp $strTemplateImageCheckbox;

	# Template-Elemente abhngig vom Login-Status:
	if($gbLoggedIn)
	{
		$strLoginForm = '';
	}
	else
	{
		$strTemplateAdminControls = '';
		$strTemplateImageCheckbox = '';
		$strLogoutForm = '';
	}

	my $nDeleteImageCount = 0;

	my $nImageCount = 0;
	my $nGroupCount = 0;

	# Statistik-Arrays
	my %nGroupImages;
	my %nGroupFromSeconds;
	my %nGroupToSeconds;
	my %nGroupTimespans;

	# Group-Links
	my $strGroupLinks;
	my %nGroupPages;

	if(opendir(HANDLE_ALBUM_ROOT, "$gImageRoot/$strAlbumName"))
	{
		my @ImageFileNames;

		if($bSortByDate)
		{
			if($bSortOrderReverse)
			{
				@ImageFileNames = sort { -M "$gImageRoot/$strAlbumName/$a" <=> -M "$gImageRoot/$strAlbumName/$b" } readdir(HANDLE_ALBUM_ROOT);
			}
			else
			{
				@ImageFileNames = sort { -M "$gImageRoot/$strAlbumName/$b" <=> -M "$gImageRoot/$strAlbumName/$a" } readdir(HANDLE_ALBUM_ROOT);
			}
		}
		else
		{
			@ImageFileNames = sort { $bSortOrderReverse ? ($b cmp $a) : ($a cmp $b) } readdir(HANDLE_ALBUM_ROOT);
		}

		# Hilfsvariable zur Berechnung der Zeitspanne fr ein Bild
		my $nSeconds;

		for my $strImageFileName (@ImageFileNames)
		{
			if($strImageFileName =~ /.jpg$|.gif$|.png$/)
			{
				$nDeleteImageCount++;

				next if('del' eq $a_mode && md5_hex($strImageFileName) eq $gSelectedImages{$nDeleteImageCount} && 1 == unlink("$gImageRoot/$strAlbumName/$strImageFileName") && ($gbVotedImagesUpdated = 1));

				$nImageCount++;

				VoteForImage("$strAlbumName/$strImageFileName") if('vote' eq $a_mode && md5_hex($strImageFileName) eq $gSelectedImages{$nImageCount});

				my $bGruppenwechsel;
				if($gbGroup)
				{
					$bGruppenwechsel = IsGruppenwechsel($strImageFileName, "$gImageDir/$strAlbumName");

					$nGroupCount++ if($nGroupCount == 0 || $bGruppenwechsel);

					# Anzahl Bilder fr aktuelle Gruppe aktualisieren
					$nGroupImages{$nGroupCount}++;

					# Zeitspanne fr aktuelle Gruppe aktualisieren
					if($bGruppenwechsel)
					{
						$nGroupFromSeconds{$nGroupCount} = $gnGroupLastSeconds;
						$nGroupPages{$nGroupCount} = int(($nImageCount - 1) / $gImagesPerPage) + 1;
					}
					else
					{
						$nGroupTimespans{$nGroupCount} += (abs($gnGroupLastSeconds - $nSeconds));
					}
					$nGroupToSeconds{$nGroupCount} = $gnGroupLastSeconds;
					$nSeconds = $gnGroupLastSeconds;
				}

				next if($nImageCount < $nPage * $gImagesPerPage + 1);
				next if($nImageCount > $nPage * $gImagesPerPage + $gImagesPerPage);

				if($gbGroup && ('' eq $strImageZeilen || $bGruppenwechsel))
				{
					# Variablen ersetzen
					my $strCurrentTemplateGroup = $strTemplateGroup;
					$strCurrentTemplateGroup = ReplaceLine($strCurrentTemplateGroup, '@@GROUP@@', $nGroupCount);
					$strCurrentTemplateGroup = ReplaceLine($strCurrentTemplateGroup, '@@GROUP_IMAGES@@', sprintf('@@GROUP%d_IMAGES@@', $nGroupCount));
					$strCurrentTemplateGroup = ReplaceLine($strCurrentTemplateGroup, '@@GROUP_FROM@@', sprintf('@@GROUP%d_FROM@@', $nGroupCount));
					$strCurrentTemplateGroup = ReplaceLine($strCurrentTemplateGroup, '@@GROUP_TO@@', sprintf('@@GROUP%d_TO@@', $nGroupCount));
					$strCurrentTemplateGroup = ReplaceLine($strCurrentTemplateGroup, '@@GROUP_TIMESPAN@@', sprintf('@@GROUP%d_TIMESPAN@@', $nGroupCount));
					$strCurrentTemplateGroup = ReplaceLine($strCurrentTemplateGroup, '@@GROUP_PREV@@', sprintf('@@GROUP%d_PREV@@', $nGroupCount));
					$strCurrentTemplateGroup = ReplaceLine($strCurrentTemplateGroup, '@@GROUP_NEXT@@', sprintf('@@GROUP%d_NEXT@@', $nGroupCount));

					$strImageZeilen .= $strCurrentTemplateGroup;
				}

				# Variablen ersetzen
				my $strCurrentTemplateImage = $strTemplateImage;
				$strCurrentTemplateImage = ReplaceLine($strCurrentTemplateImage, '@@IMAGE_CHECKBOX@@', $strTemplateImageCheckbox);
				$strCurrentTemplateImage = ReplaceLine($strCurrentTemplateImage, '@@IMAGE_NUMBER@@', $nImageCount);
				$strCurrentTemplateImage = ReplaceLine($strCurrentTemplateImage, '@@IMAGE_URL@@', "$gImageDir/$strAlbumName/$strImageFileName");
				$strCurrentTemplateImage = ReplaceLine($strCurrentTemplateImage, '@@IMAGE_FILENAME@@', "$strImageFileName");
				$strCurrentTemplateImage = ReplaceLine($strCurrentTemplateImage, '@@IMAGE_FILENAME_MD5@@', md5_hex($strImageFileName));
				$strCurrentTemplateImage = ReplaceLine($strCurrentTemplateImage, '@@IMAGE_VOTES@@', GetVotesForImage("$strAlbumName/$strImageFileName"));

				$strImageZeilen .= $strCurrentTemplateImage;
			}
		}
	}

	# Versteckte Formular-Variablen fr Steuerelemente (Buttons, Checkboxen, ...)
	my $htmlHiddenParameters;
	$htmlHiddenParameters .= "<input type=\"hidden\" name=\"print\" value=\"$FORM{'print'}\">" if('' ne $FORM{'print'});
	$htmlHiddenParameters .= "<input type=\"hidden\" name=\"album\" value=\"$FORM{'album'}\">" if('' ne $FORM{'album'});
	$htmlHiddenParameters .= "<input type=\"hidden\" name=\"sort\" value=\"$FORM{'sort'}\">" if('' ne $FORM{'sort'});
	$htmlHiddenParameters .= "<input type=\"hidden\" name=\"order\" value=\"$FORM{'order'}\">" if('' ne $FORM{'order'});
	$htmlHiddenParameters .= "<input type=\"hidden\" name=\"group\" value=\"$FORM{'group'}\">" if('' ne $FORM{'group'});
	$htmlHiddenParameters .= "<input type=\"hidden\" name=\"pattern\" value=\"$FORM{'pattern'}\">" if('' ne $FORM{'pattern'});
	$htmlHiddenParameters .= "<input type=\"hidden\" name=\"distance\" value=\"$FORM{'distance'}\">" if('' ne $FORM{'distance'});
	$htmlHiddenParameters .= "<input type=\"hidden\" name=\"page\" value=\"$FORM{'page'}\">" if('' ne $FORM{'page'});

	# Formular-Variablen fr Links zum Blttern

	my $urlParameters;
	$urlParameters .= "&print=$FORM{'print'}" if('' ne $FORM{'print'});
	$urlParameters .= "&album=$FORM{'album'}" if('' ne $FORM{'album'});
	$urlParameters .= "&sort=$FORM{'sort'}" if('' ne $FORM{'sort'});
	$urlParameters .= "&order=$FORM{'order'}" if('' ne $FORM{'order'});
	$urlParameters .= "&group=$FORM{'group'}" if('' ne $FORM{'group'});
	$urlParameters .= "&pattern=$FORM{'pattern'}" if('' ne $FORM{'pattern'});
	$urlParameters .= "&distance=$FORM{'distance'}" if('' ne $FORM{'distance'});

	# Links zum Blttern

	my $strPageLinks;

	my $nNumPages = int(($nImageCount + $gImagesPerPage - 1) / $gImagesPerPage);

	if($nNumPages > 1)
	{
		# "Zurck"
		if($nPage == 0)
		{
			$strPageLinks .= "[&lt;&lt;] ";
		}
		else
		{
			$strPageLinks .= "[<a href=\"$gScriptName?$urlParameters&page=" . $nPage . "\">&lt;&lt;</a>] ";
		}

		# Seitenzahlen
		for(my $nCurrentPage = 0; $nCurrentPage < $nNumPages; $nCurrentPage++)
		{
			if($nCurrentPage == $nPage)
			{
				$strPageLinks .= "[" . ($nCurrentPage + 1) . "] ";
			}
			else
			{
				$strPageLinks .= "[<a href=\"$gScriptName?$urlParameters&page=" . ($nCurrentPage + 1) . "\">" . ($nCurrentPage + 1) . "</a>] ";
			}
		}

		# "Weiter"
		if($nPage >= $nNumPages - 1)
		{
			$strPageLinks .= "[&gt;&gt;] ";
		}
		else
		{
			$strPageLinks .= "[<a href=\"$gScriptName?$urlParameters&page=" . ($nPage + 2) . "\">&gt;&gt;</a>] ";
		}
	}

	# Variablen ersetzen
	@TemplateZeilen = ReplaceLines(\@TemplateZeilen, '@@IMAGE@@', "$strImageZeilen");
	@TemplateZeilen = ReplaceLines(\@TemplateZeilen, '@@GROUPS@@', $nGroupCount);
	@TemplateZeilen = ReplaceLines(\@TemplateZeilen, '@@ALBUM_NAME@@', "$strAlbumName");
	@TemplateZeilen = ReplaceLines(\@TemplateZeilen, '@@IMAGE_COUNT@@', "$nImageCount");
	@TemplateZeilen = ReplaceLines(\@TemplateZeilen, '@@PAGE@@', $nPage + 1);
	@TemplateZeilen = ReplaceLines(\@TemplateZeilen, '@@PAGES@@', "$nNumPages");
	@TemplateZeilen = ReplaceLines(\@TemplateZeilen, '@@PAGE_LINKS@@', "$strPageLinks");
	@TemplateZeilen = ReplaceLines(\@TemplateZeilen, '@@BR_PAGE_LINKS@@', '' ne $strPageLinks ? "<br>$strPageLinks" : '');
	@TemplateZeilen = ReplaceLines(\@TemplateZeilen, '@@NOGROUP@@', "$strTemplateNoGroup");
	@TemplateZeilen = ReplaceLines(\@TemplateZeilen, '@@ADMIN_CONTROLS@@', "$strTemplateAdminControls");
	@TemplateZeilen = ReplaceLines(\@TemplateZeilen, '@@LOGOUT_FORM@@', "$strLogoutForm");
	@TemplateZeilen = ReplaceLines(\@TemplateZeilen, '@@LOGIN_FORM@@', "$strLoginForm");
	@TemplateZeilen = ReplaceLines(\@TemplateZeilen, '@@SCRIPT_URL@@', "$gScriptName");
	@TemplateZeilen = ReplaceLines(\@TemplateZeilen, '@@USERNAME@@', "$gUserName");
	@TemplateZeilen = ReplaceLines(\@TemplateZeilen, '@@HIDDEN_PARAMETERS@@', "$htmlHiddenParameters");

	if($gbGroup)
	{
		for(my $nGroup = 1; $nGroup <= $nGroupCount; $nGroup++)
		{
			my $strFromTime = FormatDateTime($nGroupFromSeconds{$nGroup}, $nGroupToSeconds{$nGroup});
			my $strToTime = FormatDateTime($nGroupToSeconds{$nGroup}, $nGroupFromSeconds{$nGroup});
			my $strTimeSpan = FormatTimeSpan($nGroupTimespans{$nGroup});

			@TemplateZeilen = ReplaceLines(\@TemplateZeilen, sprintf('@@GROUP%d_IMAGES@@', $nGroup), $nGroupImages{$nGroup});
			@TemplateZeilen = ReplaceLines(\@TemplateZeilen, sprintf('@@GROUP%d_FROM@@', $nGroup), $strFromTime);
			@TemplateZeilen = ReplaceLines(\@TemplateZeilen, sprintf('@@GROUP%d_TO@@', $nGroup), $strToTime);
			@TemplateZeilen = ReplaceLines(\@TemplateZeilen, sprintf('@@GROUP%d_TIMESPAN@@', $nGroup), $strTimeSpan);
			@TemplateZeilen = ReplaceLines(\@TemplateZeilen, sprintf('@@GROUP%d_PREV@@', $nGroup), $nGroup == 1            ? "&lt;&lt;" : "<a href=\"" . ($nPage + 1 == $nGroupPages{$nGroup - 1} ? '' : "$gScriptName?$urlParameters&page=" . $nGroupPages{$nGroup - 1}) . "#" . ($nGroup - 1) . "\">&lt;&lt;</a>");
			@TemplateZeilen = ReplaceLines(\@TemplateZeilen, sprintf('@@GROUP%d_NEXT@@', $nGroup), $nGroup == $nGroupCount ? "&gt;&gt;" : "<a href=\"" . ($nPage + 1 == $nGroupPages{$nGroup + 1} ? '' : "$gScriptName?$urlParameters&page=" . $nGroupPages{$nGroup + 1}) . "#" . ($nGroup + 1) . "\">&gt;&gt;</a>");
			$strGroupLinks .= "[<a href=\"" . ($nPage + 1 == $nGroupPages{$nGroup} ? '' : "$gScriptName?$urlParameters&page=" . $nGroupPages{$nGroup}) . "#$nGroup\" title=\"$strFromTime - $strToTime  ($strTimeSpan)\">$nGroup</a>] ";
		}
	}

	if($nGroupCount > 0)
	{
		$strTemplateGroupPageHeader = ReplaceLine($strTemplateGroupPageHeader, '@@GROUP_COUNT@@', $nGroupCount);
		$strTemplateGroupPageHeader = ReplaceLine($strTemplateGroupPageHeader, '@@ALBUM_FROM@@', FormatDateTime($nGroupFromSeconds{1}, $nGroupToSeconds{$nGroupCount}));
		$strTemplateGroupPageHeader = ReplaceLine($strTemplateGroupPageHeader, '@@ALBUM_TO@@', FormatDateTime($nGroupToSeconds{$nGroupCount}, $nGroupFromSeconds{1}));
		$strTemplateGroupPageHeader = ReplaceLine($strTemplateGroupPageHeader, '@@GROUP_LINKS@@', $strGroupLinks);
	}
	else
	{
		$strTemplateGroupPageHeader = '';
	}

	@TemplateZeilen = ReplaceLines(\@TemplateZeilen, '@@GROUP_PAGEHEADER@@', "$strTemplateGroupPageHeader");

	# Die HTML-Daten ausgeben
	print @TemplateZeilen;

	# Bildbewertungen schreiben
	SaveImageVotes();
}

#
# bertrgt die mittels "POST" an das Formular bergebenen Daten
# in den Hash, dessen Referenz der Funktion bergeben wird. Der
# Zugriff auf die Daten erfolgt dann mit $Hashname{'Feldname'}.
#
sub ParseFormData
{
	my $FormHashRef = shift;
	my $ImageHashRef = shift;

	my $formdata;

	if('GET' eq $ENV{'REQUEST_METHOD'})
	{
		$formdata = $ENV{'QUERY_STRING'};
	}
	else
	{
		read(STDIN, $formdata, $ENV{'CONTENT_LENGTH'});
	}

	my @pairs = split(/&/, $formdata);

	for my $pair (@pairs)
	{
		my ($name, $value) = split(/=/, $pair);
		$value =~ tr/+/ /;
		$value =~ s/%([a-fA-F0-9][a-fA-F0-9])/pack("C", hex($1))/meg;

		# Die Werte der gecheckten Image-Checkboxen werden in ein spezielles
		# Hash geschrieben mit der Bildnummer als Key und dem MD5-Wert des
		# Dateinamens als Wert:

		if('image' eq "$name")
		{
			my ($strImageNumber, $strImageFilenameMD5) = split(/\|/, $value);
			if('' ne $strImageNumber && '' ne $strImageFilenameMD5)
			{
				${$ImageHashRef}{$strImageNumber} = $strImageFilenameMD5;
			}
		}
		else
		{
			${$FormHashRef}{$name} = $value;
		}
	}
}

#
# bertrgt die vom Webserver an das Script bergebenen Cookies
# in den Hash, dessen Referenz der Funktion bergeben wird. Der
# Zugriff auf die Daten erfolgt dann mit $Hashname{'Cookiename'}.
#

sub ParseCookies
{
	my $CookiesHashRef = shift;

	my @cookies = split(/[;,]\s*/, $ENV{'HTTP_COOKIE'});

	for my $pair (@cookies)
	{
		my ($name, $value) = split(/=/, $pair);
		$value =~ tr/+/ /;
		$value =~ s/%([a-fA-F0-9][a-fA-F0-9])/pack("C", hex($1))/eg;
		${$CookiesHashRef}{$name} = $value;
	}
}

#
# Ausgabe des HTTP-Headers mit Cookies incl. Authentifizierung des Logins mit Benutzername und Pawort
#
sub PrintHttpHeader
{
	#### Beginn HTTP-Header ####

	# Cache-Control: Der Proxy soll doch bitteschn neue Seiten holen!

	print "Cache-Control: no-cache\n";
	print "Pragma: no-cache\n";

	# HTML-Entsprechung fr Cache-Control:
	#
	# <meta http-equiv="cache-control" content="no-cache">
	# <meta http-equiv="pragma" content="no-cache">

	#### Authentifizierung ####

	my $LoginAblaufdatum = MakeCookieExpireDatum(3600);	# Das Login soll nach 1 Stunde Inaktivitt ablaufen.

	# Der Benutzer meldet sich ab:
	if($FORM{'logout'})
	{
		DeleteCookie($gCookiePrefix . 'Username');
		DeleteCookie($gCookiePrefix . 'Password');
	}
	# Der Benutzer meldet sich an:
	elsif($FORM{'login'})
	{
		my $username = $FORM{'username'};
		my $password = $FORM{'password'};

		if(VerifyLogin($username, $password))
		{
			SetCookie($gCookiePrefix . 'Username', $username, $LoginAblaufdatum);
			SetCookie($gCookiePrefix . 'Password', md5_hex($password), $LoginAblaufdatum);
		}
	}
	# Prfen, ob der Benutzer bereits angemeldet ist:
	else
	{
		my $username = $gCookies{$gCookiePrefix . 'Username'};
		my $password = $gCookies{$gCookiePrefix . 'Password'};

		if('' ne $username && '' ne $password)
		{
			if(VerifyLogin($username, $password, 1))
			{
				# Login-Cookies "auffrischen"
				SetCookie($gCookiePrefix . 'Username', $username, $LoginAblaufdatum);
				SetCookie($gCookiePrefix . 'Password', $password, $LoginAblaufdatum);
			}
			else
			{
				DeleteCookie($gCookiePrefix . 'Username');
				DeleteCookie($gCookiePrefix . 'Password');
			}
		}
	}

	print "Content-type: text/html; charset=iso-8859-1\n\n";

	#### Ende HTTP-Header ####

#DEBUG:
#print "FORM: ";
#for my $name (keys %FORM) { printf("%s=\"%s\", ", $name, $FORM{$name}); }
#print "<p>$LoginAblaufdatum";

}

#
# Gibt den HTTP-Header-Code zum Setzen eines Cookies aus.
# Parameter 1: Name des Cookies, Parameter 2: Wert des Cookies,
# Parameter 3 (optional): Ablaufdatum des Cookies in Unix-Schreibweise
# oder '0' fr Session-Ende. Wenn nicht angegeben, unbegrenzt gltig.
#

sub SetCookie
{
	my $CookieName = shift;
	my $CookieValue = shift;
	my $CookieExpireDate = shift;

	$CookieValue =~ s/([^A-Za-z0-9])/sprintf("%%%02X", ord($1))/seg;
	$CookieExpireDate = 'Fri, 01-Jan-2038 00:00:00 GMT' if('' eq $CookieExpireDate);

	if('0' eq $CookieExpireDate)
	{
		print "Set-Cookie: $CookieName=$CookieValue\n";
	}
	else
	{
		print "Set-Cookie: $CookieName=$CookieValue; Expires=$CookieExpireDate\n";
	}
}

#
# Gibt den HTTP-Header-Code zum Lschen eines Cookies aus.
# Parameter 1: Name des Cookies
#

sub DeleteCookie
{
	my $CookieName = shift;

	print "Set-Cookie: $CookieName=; Expires=Thu, 01-Jan-1970 00:00:01 GMT\n";
}

#
# Erzeugt ein Expire-Datum fr Cookies, das um die Anzahl von Sekunden in der Zukunft
# liegt, die im 1. Parameter angegeben sind.
#

sub MakeCookieExpireDatum
{
	my $Sekunden = shift;

	my @Wochentage = ('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat');
	my @Monate = ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec');
	my ($Sekunde, $Minute, $Stunde, $Tag, $Monat, $Jahr, $Wochentag) = gmtime(time + $Sekunden);
	my $ExpireDate = sprintf("%s, %02d-%s-%04d %02d:%02d:%02d GMT", $Wochentage[$Wochentag], $Tag, $Monate[$Monat], $Jahr + 1900, $Stunde, $Minute, $Sekunde);

	$ExpireDate;
}

#
# berprfung des Benutzers (Parameter 1) und des Pawortes (Parameter 2) auf Gltigkeit.
# Falls das bergebene Pawort crypted ist, mu als 3. Parameter ein true-Wert bergeben werden.
# Wenn die berprfung erfolgreich war, wird $gbLoggedIn = 1 gesetzt und der Benutzername in
# $gUserName gespeichert; der Rckgabewert ist dann true (andernfalls false).
#

sub VerifyLogin
{
	my $User = shift;
	my $Password = shift;
	my $bCrypted = shift;

	my $CryptedPassword = ($bCrypted) ? $Password : md5_hex($Password);

	$gbLoggedIn = 0;
	$gUserName = '';

	if('' ne $User && '' ne $Password && $gUsers{$User} eq $CryptedPassword)
	{
		$gbLoggedIn = 1;
		$gUserName = $User;
	}

	$gbLoggedIn;
}

sub ReplaceLines
{
	my $refZeilen = shift;
	my $strSearch = shift;
	my $strReplace = shift;

	my @ReturnZeilen = @{$refZeilen};

	for my $Zeile (@ReturnZeilen)
	{
		$Zeile =~ s/$strSearch/$strReplace/g;
	}
	
	@ReturnZeilen;
}

sub ReplaceLine
{
	my $strZeile = shift;
	my $strSearch = shift;
	my $strReplace = shift;

	my $ReturnZeile = $strZeile;

	$ReturnZeile =~ s/$strSearch/$strReplace/g;
	
	$ReturnZeile;
}

# Extrahiert aus dem referenzierten Zeilen-Array den Teil zwischen dem Start- und dem End-String
# und ersetzt diesen Teil im Zeilen-Array anschlieend durch den Ersetzungsstring.

sub ExtractAndReplaceLines
{
	my $refZeilen = shift;
	my $strStart = shift;
	my $strEnd = shift;
	my $strReplace = shift;

	my @ZeilenNeu;
	my $strReturn;

	my $bFound = 0;
	my $bOccurrences = 0;

	for my $Zeile (@{$refZeilen})
	{
		if($bFound)
		{
			if($Zeile =~ /$strEnd/)
			{
				$bFound = 0;
			}
			else
			{
				$strReturn .= $Zeile if($bOccurrences == 1);
			}
		}
		else
		{
			if($Zeile =~ /$strStart/)
			{
				$bFound = 1;
				$bOccurrences++;
				push @ZeilenNeu, $strReplace;
			}
			else
			{
				push @ZeilenNeu, $Zeile;
			}
		}
	}

	@{$refZeilen} = @ZeilenNeu;

	$strReturn;
}

#
# Liest die aktuellen Bildbewertungen aus der Datei in den Hash ein.
#
sub ReadImageVotes
{
	my $nCountVotedImages = 0;

	if('' ne "$gFileNameImageVotes")
	{
		if(open(IMAGE_VOTES, "$gFileNameImageVotes"))
		{
#			if(flock(IMAGE_VOTES, LOCK_EX))
			{
				my @Zeilen = <IMAGE_VOTES>;
				close(IMAGE_VOTES);

				$nCountVotedImages = @Zeilen;

				for my $Zeile (@Zeilen)
				{
					chomp $Zeile;
					my ($nVotes, $strImageRelPath) = split(/\|/, $Zeile);
					$gVotedImages{$strImageRelPath} = $nVotes;
				}
			}
#			else
#			{
#				$gbVotedImagesReadError = 1;
#			}
		}
		else
		{
			$gbVotedImagesReadError = 1;
		}
	}

	$nCountVotedImages;
}

#
# Schreibt die aktuellen und die neuen Bildbewertungen in die Datei zurck.
#
sub SaveImageVotes
{
	if('' ne "$gFileNameImageVotes" && $gbVotedImagesUpdated && !$gbVotedImagesReadError)
	{
		for my $strImageRelPath (keys %gNewVotedImages)
		{
			$gVotedImages{$strImageRelPath} += $gNewVotedImages{$strImageRelPath};
		}

		my @Zeilen;

		for my $strImageRelPath (keys %gVotedImages)
		{
			if( -e "$gImageRoot/$strImageRelPath")
			{
				push @Zeilen, "$gVotedImages{$strImageRelPath}|$strImageRelPath\n";
			}
		}

		BackupImageVotes();

		if(open(IMAGE_VOTES, ">$gFileNameImageVotes"))
		{
			if(flock(IMAGE_VOTES, LOCK_EX))
			{
				print IMAGE_VOTES @Zeilen;
				close(IMAGE_VOTES);
			}
		}
	}
}

#
# Vergibt eine Stimme fr das angegebene Bild.
#
sub VoteForImage
{
	my $strImageRelPath = shift;

	if('' ne "$gFileNameImageVotes" && !$gbVotedImagesReadError)
	{
		$gNewVotedImages{$strImageRelPath}++;
		$gbVotedImagesUpdated = 1;
	}
}

#
# Vergleichsfunktion zum Sortieren der in der Vote-Datei enthaltenen Zeilen
# nach Anzahl der Stimmen (absteigend, d.h. die Eintrge mit den meisten Stimmen stehen oben).
#

sub SortVotedImages
{
	my $a = shift;
	my $b = shift;

	my ($nVotes_a, $strImageRelPath_a) = split(/\|/, $a);
	my ($nVotes_b, $strImageRelPath_b) = split(/\|/, $b);

	my $strCompare_a = sprintf("%08d%s", $nVotes_a, $strImageRelPath_a);
	my $strCompare_b = sprintf("%08d%s", $nVotes_b, $strImageRelPath_b);

	$strCompare_b cmp $strCompare_a ;
}

#
# Ermittelt die Anzahl der abgegebenen Stimmen fr das angegebene Bild.
#
sub GetVotesForImage
{
	my $strImageRelPath = shift;

	my $nVotes = 0;

	if('' ne "$gFileNameImageVotes")
	{
		$nVotes = $gVotedImages{$strImageRelPath} + $gNewVotedImages{$strImageRelPath};
	}

	$nVotes;
}

#
# Tagesbackup der Vote-Datei erzeugen, falls noch nicht vorhanden!
#
#

sub BackupImageVotes()
{
	if('' ne "$gFileNameImageVotesBackupFormat")
	{
		my $strFileNameImageVotesBackup = sprintf($gFileNameImageVotesBackupFormat, GetDateToday());

		unless( -e "$strFileNameImageVotesBackup")
		{
			copy($gFileNameImageVotes, $strFileNameImageVotesBackup);
		}
	}
}

#
# Ermittelt, ob ein Gruppenwechsel fr den bergebenen String vorliegt.
# Dies ist dann der Fall, wenn sein ermittelter Sekundenwert grer oder gleich
# dem Gruppierungs-Differenzwert ist. Voraussetzung fr das korrekte
# Funkionieren ist eine aufsteigende Sortierung der Strings!
#
sub IsGruppenwechsel
{
	my $strFileName = shift;
	my $strDirName = shift;
	
	my $bGruppenwechsel = 0;
	
	if($gbGroup)	# nur wenn 'group'-Parameter angegeben
	{
		my $nSeconds = GetSecondsFromFilename($strFileName, $strDirName);
#print "$strFileName: nSeconds = $nSeconds, diff = " . ($nSeconds - $gnGroupLastSeconds) . ", ";
		$bGruppenwechsel = 1 if(abs($nSeconds - $gnGroupLastSeconds) >= $gnGroupSeconds);
		$gnGroupLastSeconds = $nSeconds;

#print "bGruppenwechsel = $bGruppenwechsel<br>";
		$bGruppenwechsel;
	}
}

sub GetSecondsFromFilename
{
	my $strFileName = shift;
	my $strDirName = shift;
	
	my $nSeconds = 0;

	if('date' eq $FORM{'sort'})
	{
		# Gruppierung nach Dateidatum
		$nSeconds = (stat("$gDocumentRoot/$strDirName/$strFileName"))[9];
#print "$strDirName/$strFileName: nSeconds = $nSeconds<br>";
	}
	else
	{
		# Gruppierung nach Zeitstempel im Dateinamen

		my $nLengthName = length($strFileName);
		my $nLengthPattern = length($gstrGroupPattern);

		my ($sec, $min, $hour, $mday, $mon, $year);

		# Wert aus String parsen
		for(my $nIndex = 0; $nIndex < $nLengthName && $nIndex < $nLengthPattern; $nIndex++)
		{
			my $chPattern = substr($gstrGroupPattern, $nIndex, 1);
			my $chFileName = substr($strFileName, $nIndex, 1);

			$sec .= $chFileName if('s' eq $chPattern);
			$min .= $chFileName if('n' eq $chPattern);
			$hour .= $chFileName if('h' eq $chPattern);
			$mday .= $chFileName if('d' eq $chPattern);
			$mon .= $chFileName if('m' eq $chPattern);
			$year .= $chFileName if('y' eq $chPattern);
		}

		{
			# Werte validieren
			last if($sec < 0 || $sec > 59);
			last if($min < 0 || $min > 59);
			last if($hour < 0 || $hour > 23);
			$year += 2000 if($year >= 0 && $year <= 99);
			last if(!CheckDatum($mday, $mon, $year));

			$nSeconds = timelocal($sec, $min, $hour, $mday, $mon - 1, $year - 1900);
		}
	}
	
	$nSeconds;
}

#
# berprft ein Datum auf Gltigkeit und gibt true zurck, wenn das Datum gltig ist,
# sonst false. Dies ist deshalb erforderlich, weil die Perl-Funktion timelocal()
# bescheuerterweise gleich das komplette Script abbricht, wenn das bergebene Datum
# ungltig ist, ohne da man diesen Abbruch irgendwie abfangen knnte.
#
sub CheckDatum
{
	my $tag = shift;
	my $monat = shift;
	my $jahr = shift;

	$tag += 0;
	$monat += 0;
	$jahr += 0;

	my @MaxDay = (31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31);

	return 0 if($jahr < 1902 || $jahr > 2038);
	return 0 if($monat < 1 || $monat > 12);
	return 0 if($tag < 1 || $tag > $MaxDay[$monat - 1]);
	return 0 if($tag == 29 && $monat == 2 && ($jahr % 4) != 0);
	return 1;
}

#
# Wandelt ein Anzahl von Sekunden in einen Zeitstring des Formats HH:MM:SS um
#
sub FormatTimeSpan
{
	my $nSeconds = shift;

	my $nHours = int($nSeconds / 3600); 
	$nSeconds -= $nHours * 3600;

	my $nMinutes = int($nSeconds / 60); 
	$nSeconds -= $nMinutes * 60;

	sprintf("%02d:%02d:%02d", $nHours, $nMinutes, $nSeconds);
}

#
# Wandelt einen Sekundenwert (localtime) in einen Datum-/Zeitstring des Formats [DD.MM.YYYY] HH:MM:SS um
# Der Datums-Teil wird nur ausgegeben, wenn sich das Datum des 2. Sekundenwertes vom Datum des 1. Sekundenwertes unterscheidet.
#
sub FormatDateTime
{
	my $nSeconds = shift;
	my $nSecondsCompare = shift;

	my ($nSecond, $nMinute, $nHour, $nDay, $nMonth, $nYear) = (localtime($nSeconds))[0..5];

	my $strDate = sprintf("%02d.%02d.%04d ", $nDay, $nMonth + 1, $nYear + 1900);
	my $strTime = sprintf("%02d:%02d:%02d", $nHour, $nMinute, $nSecond);

	if($nSecondsCompare)
	{
		my ($nDayCompare, $nMonthCompare, $nYearCompare) = (localtime($nSecondsCompare))[3..5];
		$strDate = '' if($nYear * 10000 + $nMonth * 100 + $nDay == $nYearCompare * 10000 + $nMonthCompare * 100 + $nDayCompare);
	}

	"$strDate$strTime";
}

#
# Heutiges Datum zurckgeben (Format: "JJJJMMTT")
#

sub GetDateToday
{
	my ($tag, $monat, $jahr) = (localtime())[3..5];
	sprintf("%04d%02d%02d", $jahr + 1900, $monat + 1, $tag);
}

########
# Ende #
########
