/*************************************************************************/
/*                                                                       */
/*  Einfache Grafikroutinen fr EGA-/VGA-Karten (benutzt Interrupt 10h)  */
/*                                                                       */
/*                        (C) 1991 Stefan Bion                           */
/*                                                                       */
/*************************************************************************/

#include <dos.h>
#include <stdio.h>

#define TEXT80X25  0x0003                /* Die verschiedenen Video-Modi */
#define EGA320X200 0x000D
#define EGA640X200 0x000E
#define EGA640X350 0x0010
#define EGA640X480 0x0012
#define VGA320X200 0x0013

#define AX r.x.ax        /* Die Registerbezeichnungen fr den Int-Aufruf */
#define BX r.x.bx
#define CX r.x.cx
#define DX r.x.dx
#define AL r.h.al
#define AH r.h.ah
#define BL r.h.bl
#define BH r.h.bh
#define CL r.h.cl
#define CH r.h.ch
#define DL r.h.dl
#define DH r.h.dh

#define INT_10H int86(0x10, &r, &r)      /* Makro fr Aufruf des Int 10h */

union REGS r; /* Beinhaltet die Strukturen mit den Registerbezeichnungen */

int maxx=80,                         /* Maximale Auflsung in x-Richtung */
    maxy=25,                         /* Maximale Auflsung in y-Richtung */
    maxc=16;                    /* Anzahl Farben im aktuellen Videomodus */

/*************************************************************************/
/* Setzen des Video-Modus: Die Variablen maxx, maxy und maxc werden      */
/* initialisiert und knnen abgefragt werden.                            */
/*************************************************************************/
void setvidmode(int v)
{
  AX=v, INT_10H;

  switch(v)
  {
    case TEXT80X25:  maxx=80;  maxy=25;  maxc=16;  break;
    case EGA320X200: maxx=320; maxy=200; maxc=16;  break;
    case EGA640X200: maxx=640; maxy=200; maxc=16;  break;
    case EGA640X350: maxx=640; maxy=350; maxc=16;  break;
    case EGA640X480: maxx=640; maxy=480; maxc=16;  break;
    case VGA320X200: maxx=320; maxy=200; maxc=256; break;
  }
}

/*************************************************************************/
/* Lscht den Bildschirm                                                 */
/*************************************************************************/
void clrscreen (void)
{
  AH=0x0F, INT_10H, AH=0x00, INT_10H;
}

/*************************************************************************/
/* Setzt ein Pixel an die Stelle x,y mit der Farbe c                     */
/*************************************************************************/
void putpixel(int x, int y, int c)
{
  int curs;

  if(maxx==80)
  {
    AH=3, BH=0, INT_10H, curs=DX;
    AH=2, BH=0, DL=x, DH=y, INT_10H;
    AH=9, AL='', BH=0, BL=c, CX=1, INT_10H;
    AH=2, BH=0, DX=curs, INT_10H;
  }
  else
  {
    AH=0x0C, AL=c, CX=x, DX=y, INT_10H;
  }
}

/*************************************************************************/
/* Liest die Farbe eines Pixels an der Stelle x,y                        */
/*************************************************************************/
unsigned char getpixel(int x, int y)
{
  int curs;

  if(maxx==80)
  {
    AH=3, BH=0, INT_10H, curs=DX;
    AH=2, BH=0, DL=x, DH=y, INT_10H;
    AH=8, BH=0, INT_10H;
    AH=2, BH=0, DX=curs, INT_10H;
    return(AH);
  }
  else
  {
    AH=0x0D, CX=x, DX=y, INT_10H;
    return(AL);
  }
}

/*************************************************************************/
/* Fllt ein Bildschirmfenster (x1, x2, y1, y2, c)                       */
/*************************************************************************/
void fillrect(int x1, int x2, int y1, int y2, int c)
{
  int x, y;

  for(y=y1; y<=y2; y++)
  {
    for(x=x1; x<=x2; x++) putpixel(x, y, c);
  }
}

/*************************************************************************/
/* Zeichnet eine Linie mit der Farbe c von x1/y1 nach x2/y2              */
/*************************************************************************/
void drawline(int x1, int y1, int x2, int y2, int c)
{
  int x, y, h, dx,dy;

  if(x1>x2) h=x2, x2=x1, x1=h;
  if(y1>y2) h=y2, y2=y1, y1=h;

  if(x1==x2)
  {
    if(y1==y2) putpixel(x1,y1,c);
    else
    {
      if(y1>y2) h=y2, y2=y1, y1=h;
      for(y=y1;y<=y2;y++) putpixel(x1,y,c);
    }
    return;
  }

  if(y1==y2)
  {
    if(x1==x2) putpixel(x1,y1,c);
    else
    {
      if(x1>x2) h=x2, x2=x1, x1=h;
      for(x=x1;x<=x2;x++) putpixel(x,y1,c);
    }
    return;
  }

  if(x1>x2) h=x2, x2=x1, x1=h;
  dx=x2-x1+1, dy=y2-y1+1;
  for(x=0; x<dx; x++) putpixel(x1+x,y1+(long)dy*x/dx,c);
}
