;**************************************************************************
;**                                                                      **
;**   SDIGIT.ASM (C) Redaktionsbro Everts & Hagedorn DOS International  **
;**                                                                      **
;**   Maschinenroutinen zum Ausgeben und Einlesen von Sounds mit dem     **
;**   dem DOS-Digitizer:                                                 **
;**                                                                      **
;**   void getdport (void);                                              **
;**   void play (char far* block);                                       **
;**   void rep (char far* block);                                        **
;**   void listen (char rate);                                           **
;**   void rec (char far* block);                                        **
;**                                                                      **
;**   Angepat zum Einbinden in C-Programme von Stefan Bion.             **
;**   Soll ein anderes als das Speichermodell SMALL verwendet werden,    **
;**   dann ist die untenstehende .MODEL-Anweisung entsprechend zu        **
;**   ndern. Dieses Assemblermodul mu mit TASM assembliert werden.     **
;**                                                                      **
;**************************************************************************

                   .model large

;--------------------------------------------------------------------------
; Weil die Prozessoren V20 und V30 den HLT-Befehl ignorieren, luft das
; Programm auf Rechnern mit diesen Prozessoren nicht. Ersetzen Sie in einem
; solchen Fall im folgenden Makro den Befehl "hlt" durch "x: jmp x". Dann
; mssen Sie aber bei langsamen PCs im Lautsprecher-Modus ein leichtes
; Rauschen in Kauf nehmen.
;--------------------------------------------------------------------------
CPUHALT            macro
                   local x
                   hlt
                   endm

;--------------------------------------------------------------------------
; Wenn Sie einen anderen als den DOS-Digitizer einsetzen wollen, so mssen
; Sie nur die folgenden drei Makros ndern:
;--------------------------------------------------------------------------
; Dieses Makro prft, ob der Digitizer am Druckerport [DX] angeschlossen
; ist (CF=1) oder nicht (CF=0):
;--------------------------------------------------------------------------
DTEST              macro
                   push cx
                   push dx

                   xor al,al                ; Datenport auf 0, falls ein angeschlossener
                   out dx,al                ; Drucker Select nicht interpretiert

                   inc dx                   ; Wenn Busy, dann kein Digitizer
                   in al,dx
                   shl al,1
                   jnc nichtda

                   cli                      ; Mit der Strobe-Leitung eine Wandlung
                   inc dx                   ; auslsen, dabei Select auf 0
                   mov al,05h
                   out dx,al
                   mov al,0Ch
                   out dx,al

                   dec dx                   ; Wenn kein Busy, dann ist gar nichts
                   in al,dx                 ; angschlossen
                   sti
                   shl al,1
                   cmc
                   jnc nichtda

                   mov cx,100               ; Wenn innerhalb einer bestimmten Zeit
                                            ; der Busy-Anschlu wieder gelscht
warten:            in al,dx                 ; wird, dann ist ein Digitizer vorhanden
                   shl al,1
                   jc nichtda
                   loop warten

nichtda:           pop dx
                   pop cx
                   endm

;--------------------------------------------------------------------------
; Das nchste Makro liest ein Byte vom AD-Wandler ins AL-Register ein, dabei
; enthlt DX die Adresse des Drucker-Ports
;--------------------------------------------------------------------------
INBYTE             macro
                   inc dx                   ; Oberes Halbbyte einlesen
                   in al,dx
                   shl al,1
                   mov ah,al
                   inc dx                   ; Multiplexer umschalten
                   mov al,0Eh
                   out dx,al
                   dec dx                   ; Unteres Halbbyte einlesen
                   in al,dx
                   shr al,1
                   shr al,1
                   shr al,1
                   and ax,0F00Fh            ; Halbbytes miteinander verknpfen
                   or ah,al

                   inc dx                   ; Multiplexer zurcksetzen und
                   mov al,0Dh               ; Startimpuls fr neue Umwandlung
                   out dx,al                ; geben
                   dec ax
                   out dx,al
                   dec dx
                   dec dx
                   mov al,ah
                   endm

;--------------------------------------------------------------------------
; Dieses Makro gibt das Byte in AL auf dem Digitizer aus.
;--------------------------------------------------------------------------
OUTBYTE            macro
                   out dx,al
                   endm

;**************************************************************************
; Codesegment:
;**************************************************************************
                   .code

;--------------------------------------------------------------------------
; Variablen fr die Turbo-C-Routinen:
;--------------------------------------------------------------------------
lptport            dw 0

;--------------------------------------------------------------------------
; Lokale Variablen der Routinen _PLAY, _LISTEN und _REC:
;--------------------------------------------------------------------------
oldint08           equ word ptr [bp-14]
oldint09           equ [bp-10]
mastermask         equ byte ptr [bp-6]
slavemask          equ byte ptr [bp-5]
abbruch            equ word ptr [bp-4]
continue           equ word ptr [bp-2]

;--------------------------------------------------------------------------
; Folgende Prozedur setzt die Interrupt-Vektoren und initialisiert den
; System-Timer
;--------------------------------------------------------------------------
einschalten        proc near

                   mov dx,03F2h             ; Diskettenmotoren abschalten
                   xor al,al
                   out dx,al

                   in al,21h                ; Interrupt-Masken retten
                   mov mastermask,al
                   in al,0A1h
                   mov slavemask,al

                   mov al,0FCh              ; Interrupts bis auf Timer und
                   out 21h,al               ; Keyboard aus
                   mov al,0FFh
                   out 0A1h,al

                   cli                      ; Interrupts sperren
                   cld                      ; Stringzugriffe in aufsteigender Reihenfolge
                   xor ax,ax                ; Adresse fr Vektorzugriff setzen
                   mov es,ax

                   xchg si,es:[08h*4]       ; Interrupt-Vektor 08h retten und neu setzen
                   mov oldint08,si
                   mov ax,cs
                   xchg ax,es:[08h*4+2]
                   mov oldint08+2,ax

                   xchg di,es:[09h*4]       ; Interrupt-Vektor 09h retten und neu setzen
                   mov word ptr oldint09,di
                   mov ax,cs
                   xchg ax,es:[09h*4+2]
                   mov word ptr oldint09+2,ax

                   mov al,24h               ; Highbyte des System-Timers auf Null
                   out 43h,al
                   xor al,al
                   out 40h,al
                   mov al,14h
                   out 43h,al
                   ret

einschalten        endp

;--------------------------------------------------------------------------
; Bei jedem Drcken oder Loslassen einer Taste wird der Interrupt 09h aufgerufen
;--------------------------------------------------------------------------
neuerint09:        mov ah,al                ; Scan-Code der Taste lesen
                   in al,60h
                   and al,al                ; Wurde die Taste gedrckt?
                   jns keypressed
                   cmp al,0E0h
                   jae keypressed

                   in al,61h                ; Wenn nicht, dann ignorieren;
                   or al,80h                ; Besttigung an Tastatur und
                   out 61h,al               ; Controller geben
                   and al,7Fh
                   out 61h,al
                   mov al,61h
                   out 20h,al
                   mov al,ah
                   add sp,6                 ; Flags und Rcksprungadresse vom
                   jmp continue             ; Stapel und weitermachen

keypressed:        call abschalten          ; Taste gedrckt: Interrupts wiederherstellen
                   add sp,2                 ; Abbruchadresse auf Stack
                   push abbruch
                   jmp dword ptr oldint09   ; Tastaturtreiber aufrufen

;--------------------------------------------------------------------------
; Folgende Prozedur setzt die verbogenen Interrupt-Vektoren und den Timer
; wieder auf ihre ursprnglichen Werte
;--------------------------------------------------------------------------
abschalten         proc near

                   mov al,36h               ; Ticker-Timer auf Normalwert
                   out 43h,al
                   xor al,al
                   out 40h,al
                   out 40h,al

                   mov al,mastermask        ; Restliche Interrupts wieder zulassen
                   out 21h,al
                   mov al,slavemask
                   out 0A1h,al

                   push ss                  ; Interrupt-Vektoren wiederherstellen:
                   pop ds                   ; DS:SI := SS:oldint08
                   lea si,oldint08
                   xor ax,ax                ; ES:DI := Int-08h-Vektor in der
                   mov es,ax                ; Interruptvektor-Tabelle.
                   mov di,08h*4
                   cld
                   movsw                    ; Gesicherte Interruptvektoren
                   movsw                    ; zurckschreiben.
                   movsw
                   movsw
                   ret

abschalten         endp

;**************************************************************************
;*                                                                        *
;*    Die folgenden Prozeduren sind die Schnittstellen zu Turbo-C         *
;*                                                                        *
;**************************************************************************

;--------------------------------------------------------------------------
; Von Turbo-C aufgerufen bestimmt die folgende Prozedur die parallele
; Schnittstelle, an die der Digitizer angeschlossen ist:
;--------------------------------------------------------------------------
                   public _getdport         ; void getdport (void);

_getdport          proc
                   push si
                   push di

                   xor cx,cx                ; Ab 0:408h stehen die Portadressen der
                   mov es,cx                ; vier parallelen Schnittstellen (oder 0)
                   mov bx,0408h

testport:          mov dx,es:[bx]
                   and dx,dx                ; Wenn keine Karte vorhanden, dann
                   je noport                ; Adresse bergehen
                   mov cx,dx                ; Aktuellen Wert als hchste
                   DTEST                    ; Portadresse speichern.
                   jc dfound

noport:            inc bx                   ; Vier Ports sind zu berprfen
                   inc bx
                   cmp bx,0410h
                   jb testport
                   xor dx,dx                ; Kein Digitizer gefunden; DX := 0

dfound:            mov lptport,dx
                   and dx,dx                ; Wenn nicht gefunden, dann wird die
                   jne portok               ; hchste im System vorkomende
                   mov lptport,cx           ; Schnittstelle angenommen

portok:            pop di
                   pop si
                   ret

_getdport          endp

;--------------------------------------------------------------------------
; Block unter Turbo-C abspielen (Digitizer)
;--------------------------------------------------------------------------
                   public _play             ; void play (char far* block);

_play              proc
                   arg adr:dword

                   push bp
                   mov bp,sp
                   sub sp,14                ; Platz fr lokale Variablen schaffen
                   push ds
                   push si
                   push di

                   mov si,offset playint08  ; Digitizer einschalten
                   mov di,offset neuerint09
                   mov abbruch,offset playexit
                   mov continue,offset playcont
                   call einschalten
                   mov dx,lptport
                   les di,adr               ; ES:DI := Adresse des Blocks

playblock:         mov al,es:[di+8]         ; Abspielgeschwindigkeit des Blocks
                   out 40h,al               ; in den Ticker-Timer schreiben.
                   mov si,es:[di]           ; AX:SI := Anfangsadresse
                   mov ax,es:[di+2]
                   mov cx,es:[di+4]         ; BX:CX := Endadresse + 1
                   mov bx,es:[di+6]

playwert:          mov ds,ax                ; Prfen, ob das Blockende erreicht wurde
                   cmp ax,bx
                   jne playnext
                   cmp si,cx
                   je playlast

playnext:          lodsb                    ; Nchsten Wert lesen und auf Interrupt
                                            ; warten
playcont:          sti
                   CPUHALT

playint08:         OUTBYTE                  ; Wert ausgeben
                   add sp,6                 ; Rcksprungadresse und Flags vom Stack
                                            ; entfernen
                   mov al,60h               ; Interruptbesttigung an Controller
                   out 20h,al
                   mov ax,ds                ; Falls ntig, Segmentadresse weiterzhlen
                   and si,15                ; und nchsten Wert ausgeben
                   jne playwert
                   inc ax
                   jmp playwert

playlast:          call abschalten
                   sti

playexit:          pop di
                   pop si
                   pop ds
                   mov sp,bp                ; Platz fr lokale Variablen freigeben
                   pop bp
                   ret

_play              endp

;--------------------------------------------------------------------------
; Block unter Turbo-C abspielen (Lautsprecher)
;--------------------------------------------------------------------------
                   public _lplay            ; void lplay (char far* block);

_lplay             proc
                   arg adr:dword

                   push bp
                   mov bp,sp
                   sub sp,14                ; Platz fr lokale Variablen schaffen
                   push ds
                   push si
                   push di

                   mov si,offset lplayint08 ; Digitizer einschalten
                   mov di,offset neuerint09
                   mov abbruch,offset lplayexit
                   mov continue,offset lplaycont
                   call einschalten

                   Mov AL,66h               ; Refresh-Timer setzen
                   Out 43h,AL
                   Xor AL,AL
                   Out 41h,AL
                   Mov AL,56h
                   Out 43h,AL
                   Mov AL,54
                   Out 41h,AL
                   Mov AL,0B0h              ; Sound-Timer setzen
                   Out 43h,AL
                   Xor AL,AL
                   Out 42h,AL
                   Out 42h,AL
                   Mov AL,90h
                   Out 43h,AL
                   In AL,61h                ; Lautsprecher anschalten
                   Or AL,3
                   Out 61h,AL
                   Push CS
                   Pop ES

                   les di,adr               ; ES:DI := Adresse des Blocks

lplayblock:        mov al,es:[di+8]         ; Abspielgeschwindigkeit des Blocks
                   out 40h,al               ; in den Ticker-Timer schreiben.
                   out 41h,al
                   mov si,es:[di]           ; AX:SI := Anfangsadresse
                   mov ax,es:[di+2]
                   mov cx,es:[di+4]         ; BX:CX := Endadresse + 1
                   mov bx,es:[di+6]

lplaywert:         mov ds,ax                ; Prfen, ob das Blockende erreicht wurde
                   cmp ax,bx
                   jne lplaynext
                   cmp si,cx
                   je lplaylast

lplaynext:         lodsb                    ; Wert lesen und von 0 bis 255
                   Shr AL,1                 ; auf 1 bis 64 skalieren
                   Shr AL,1
                   Inc AX

lplaycont:         sti
                   CPUHALT

lplayint08:        Out 42h,AL               ; Wert auf dem Lautsprecher ausgeben
                   add sp,6                 ; Rcksprungadresse und Flags vom Stack
                                            ; entfernen
                   mov al,60h               ; Interruptbesttigung an Controller
                   out 20h,al
                   mov ax,ds                ; Falls ntig, Segmentadresse weiterzhlen
                   and si,15                ; und nchsten Wert ausgeben
                   jne lplaywert
                   inc ax
                   jmp lplaywert

lplaylast:         call abschalten

                   In AL,61h                ; Lautsprecher abschalten
                   And AL,254
                   Out 61h,AL
                   Mov AL,76h               ; Refresh auf Normalwert
                   Out 43h,AL
                   Mov AL,18
                   Out 41h,AL
                   Xor AL,AL
                   Out 41h,AL

                   sti

lplayexit:         pop di
                   pop si
                   pop ds
                   mov sp,bp                ; Platz fr lokale Variablen freigeben
                   pop bp
                   ret

_lplay             endp

;--------------------------------------------------------------------------
; Block unter Turbo-C wiederholt abspielen (repetieren)
;--------------------------------------------------------------------------
                   public  _rep             ; void rep (char far* block);

_rep               proc
                   arg adr:dword

                   push bp
                   mov bp,sp
                   sub sp,14                ; Platz fr lokale Variablen schaffen
                   push ds
                   push si
                   push di

                   mov si,offset repint08   ; Digitizer einschalten
                   mov di,offset neuerint09
                   mov abbruch,offset repexit
                   mov continue,offset repcont
                   call einschalten
                   mov dx,lptport
                   les di,adr               ; ES:DI := Adresse des Blocks

repblock:          mov al,es:[di+8]         ; Abspielgeschwindigkeit des Blocks
                   out 40h,al               ; in den Ticker-Timer schreiben.
repeat:            mov si,es:[di]           ; AX:SI := Anfangsadresse
                   mov ax,es:[di+2]
                   mov cx,es:[di+4]         ; BX:CX := Endadresse + 1
                   mov bx,es:[di+6]

repwert:           mov ds,ax                ; Prfen, ob das Blockende erreicht wurde
                   cmp ax,bx
                   jne repnext
                   cmp si,cx
                   je repeat

repnext:           lodsb                    ; Nchsten Wert lesen und auf Interrupt
                                            ; warten
repcont:           sti
                   CPUHALT

repint08:          OUTBYTE                  ; Wert ausgeben
                   add sp,6                 ; Rcksprungadresse und Flags vom Stack
                                            ; entfernen
                   mov al,60h               ; Interruptbesttigung an Controller
                   out 20h,al
                   mov ax,ds                ; Falls ntig, Segmentadresse weiterzhlen
                   and si,15                ; und nchsten Wert ausgeben
                   jne repwert
                   inc ax
                   jmp repwert

repexit:           pop di
                   pop si
                   pop ds
                   mov sp,bp                ; Platz fr lokale Variablen freigeben
                   pop bp
                   ret

_rep               endp

;--------------------------------------------------------------------------
; Zuhren
;--------------------------------------------------------------------------
                   public _listen           ; void listen (char rate);

_listen            proc
                   arg speed:byte:2

                   push bp
                   mov bp,sp
                   sub sp,14                ; Platz fr lokale Variablen schaffen
                   push ds
                   push si
                   push di

                   mov si,offset listint08  ; Digitizer einschalten
                   mov di,offset neuerint09
                   mov abbruch,offset listexit
                   mov continue,offset listcont
                   call einschalten
                   mov dx,lptport
                   mov al,speed             ; Taktrate in den Ticker-Timer
                   out 40h,al               ; schreiben

listcont:          sti                      ; In einer Endlosschleife (Abbruch durch
                   CPUHALT                  ; Tastendruck) Werte einlesen und sofort
                                            ; wieder ausgeben
listint08:         INBYTE
                   OUTBYTE
                   add sp,6                 ; Stack reinigen
                   mov al,60h               ; Besttigung an Controller
                   out 20h,al
                   jmp listcont

listexit:          pop di
                   pop si
                   pop ds
                   mov sp,bp                ; Platz fr lokale Variablen freigeben
                   pop bp
                   ret

_listen            endp

;--------------------------------------------------------------------------
; Aufnehmen
;--------------------------------------------------------------------------
                   public _rec              ; void rec (char far* block);

_rec               proc
                   arg adr:dword

                   push bp
                   mov bp,sp
                   sub sp,14                ; Platz fr lokale Variablen schaffen
                   push ds
                   push si
                   push di

                   mov si,offset recint08   ; Digitizer einschalten
                   mov di,offset neuerint09
                   mov abbruch,offset recexit
                   mov continue,offset reccont
                   call einschalten
                   mov dx,lptport           ; Dummy-Lesevorgang, um den Wandler
                   INBYTE                   ; zu starten
                   lds si,adr               ; DS:SI := Adresse des Blocks
                   lodsw                    ; ES:DI := Anfangsadresse
                   mov di,ax
                   lodsw
                   mov es,ax
                   lodsw                    ; BX:CX := Endadresse + 1
                   mov cx,ax
                   lodsw
                   mov bx,ax

                   lodsb                    ; Aufnahmetaktrate holen und in
                   out 40h,al               ; Ticker-Timer schreiben
                   mov si,es

recwert:           cmp si,bx                ; Auf Blockende prfen
                   jne reccont
                   cmp di,cx
                   je reclast

reccont:           sti                      ; Auf Ticker-Interrupt warten
                   CPUHALT

recint08:          INBYTE                   ; Wert lesen und zur Kontrolle ausgeben
                   OUTBYTE
                   stosb                    ; Wert speichern
                   add sp,6                 ; Flags und Rcksprungadresse vom Stack
                                            ; entfernen
                   mov al,60h               ; Besttigung an Controller
                   out 20h,al
                   and di,15                ; Segmentanteil erhhen, falls erforderlich
                   jne recwert
                   inc si
                   mov es,si
                   jmp recwert

reclast:           call abschalten
                   sti

recexit:           pop di
                   pop si
                   pop ds
                   mov sp,bp                ; Platz fr lokale Variablen freigeben
                   pop bp
                   ret

_rec               endp

;--------------------------------------------------------------------------
; Ende des Codesegments und des Programms:
;--------------------------------------------------------------------------
                   end

;--------------------------------------------------------------------------
