/*****************************************************************************
*                                                                            *
*   R86 Reassembler V1.00  (C) 1992 Stefan Bion                              *
*                                                                            *
*****************************************************************************/
#include "r86.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <dos.h>
#include <signal.h>
#include <fcntl.h>

/*****************************************************************************
Globale Variablen
*****************************************************************************/
typedef unsigned char BYTE;
typedef unsigned int  WORD;
typedef unsigned long DWORD;

extern void doswrite(int);

FILE *cf=NULL, *af=NULL;
BYTE comfile[128], asmfile[128], *ap, addrfile[128];

WORD cflen;             /* Lnge der .COM-Datei */
int  d_switch=0;        /* Schalter fr Option -d */
int  s_switch=0;        /* Flag und Zhler fr ASM-Dateien (Option -s) */
int  fpopcode=0;        /* Flag fr FWAIT-Opcode */
int  patch83=0;         /* Flag fr PATCH83-Makro */
int  pass;              /* 0 = Pass 1, 1 = Pass 2 */
WORD ip;                /* Befehlszhler */
int  weiter;            /* Flag fr "Datei-Ende" */
int  prefix_flag;       /* Flag fr "Letzter Opcode war Prefix" */
int  multiple;          /* Flag fr "Multiple Operands to PUSH/POP/INC/DEC" */
int  unget_cnt;         /* Zhler fr zurckgestellte Bytes */
BYTE unget_byte[8];     /* Die zurckgestellten Bytes (max. 7) */
int  mode, last_mode;   /* A=ASCII, B=Byte, W=Word, C=Code */
int  ascii_mode=1;      /* Wenn mode==A: Byte druckbar ? */
int  emptyline;         /* Zuletzt leere Zeile ausgegeben */
WORD linecnt;           /* Zeilenzhler fr split_file() */
int  data_len;          /* Aktuelle Lnge einer Datenzeile */
WORD dupcnt, dupval;    /* Zhler und Variable fr DUP */

#define MAX_SYM 8192    /* Maximale Anzahl Symbole */
WORD symtab[MAX_SYM];   /* Symboltabelle mit Word-Referenzen */
WORD symnum=0, symcnt=0;/* Anzahl und Zhler Eintrge in symtab */

#define MAX_ADDR 2048   /* Maximale Anzahl Adressen */
struct ADDRTAB          /* Tabelle mit Adressen und Datentypen */
{ WORD addr; int type; } addrtab[MAX_ADDR];
WORD addrnum, addrcnt;  /* Anzahl und Zhler Eintrge in addrtab */

/*****************************************************************************
Video-Routinen (schreiben direkt ins Video-RAM)
*****************************************************************************/
WORD vid_cursor, vid_segment;
BYTE vid_attrib;

#define vid_putcurs(x,y) vid_cursor=160*(y)+2*(x)

void vid_putch(BYTE c)
{
  pokeb(vid_segment, vid_cursor++, c);
  pokeb(vid_segment, vid_cursor++, vid_attrib);
}

void vid_cls(void)
{
  int i;
  for(i=0; i<2000; i++) vid_putch(' ');
  vid_cursor=0;
}

void vid_puts(BYTE *s)
{
  while(*s) vid_putch(*s++);
}

void vid_puthex(WORD z, int d)
{
  static BYTE h[17]="0123456789ABCDEF";
  int i;
  BYTE b[5];
  for(i=d-1, b[d]=0; i>=0; i--) b[i]=h[z&15], z>>=4;
  vid_puts(b);
}

/*****************************************************************************
Adresse in addrtab einfgen:
*****************************************************************************/
void addaddr(WORD start, WORD end, int type, WORD max)
{
  struct ADDRTAB helptab[MAX_ADDR];    /* Hilfstabelle */
  WORD i=0, h=0;   /* Indizes fr Original- und Hilfstabelle */

  WORD a, w=start;
  int t=C, p=0;

  for(i=0; i<addrnum; i++)
  {
    if((a=addrtab[i].addr)<w)
    {
      t=addrtab[i].type;
      if(p==0 || p==2) helptab[h].addr=a, helptab[h].type=t, h++;
    }
    else
    {
      helptab[h].addr=w; if(a==w) t=addrtab[i].type; else i--;
      switch(p) {
        case 0: helptab[h].type=type; w=end, p=1; break;
        case 1: helptab[h].type=t; w=0xFFFF, p=2; break; }
      h++;
    }
  }

  switch(p)
  {
    case 0:
      if(i) t=addrtab[i-1].type; else t=C;
      helptab[h].addr=start, helptab[h].type=type, h++;
    case 1:
      helptab[h].addr=end, helptab[h].type=t, h++;
  }

  for(i=0; i<h && helptab[i].type==C; i++);

  for(addrnum=0; i<h; i++)
  {
    if(!(addrnum &&
        (helptab[i].type==helptab[i-1].type || helptab[i].addr==max)))
    {
      addrtab[addrnum].addr=helptab[i].addr;
      addrtab[addrnum].type=helptab[i].type;
      addrnum++;
    }
  }
}

/*****************************************************************************
COM-Datei dumpen:
*****************************************************************************/
void dump(void)
{
  BYTE buffer[16][16];
  static BYTE *types[4]={"(Code) ","(Word) ","(Byte) ","(ASCII)"};
  WORD a=0, i, i1, l=cflen, n, o, x, y, xc=0, yc=0;
  BYTE c, tc=0x1F, nc=0x70, cc=0x0F, mc[4]={ 0x74, 0x71, 0x7A, 0x7E };
  WORD c_start, c_end;
  int c_type;
  WORD m_start=0, m_end=0;
  int m_type=0;

  vid_segment=(((peekb(0,0x410)>>4)&3)==3)?0xB000:0xB800;
  if(vid_segment==0xB000)
    tc=0x07, nc=0x70, cc=0x0F, mc[0]=0x70, mc[1]=0x70, mc[2]=0x70, mc[3]=0x70;

  _setcursortype(_NOCURSOR); vid_attrib=nc; vid_cls(); vid_attrib=tc;
  vid_puts("             R86 Reassembler V1.00  Copyright (c) 1992 Stefan Bion              ");

  vid_putcurs(0,24);
  vid_puts("   Current area:     -               Marked: none                               ");
  vid_putcurs(57,24); vid_puts(comfile);

  vid_putcurs(0,2); vid_attrib=nc; vid_puts("   "); vid_puthex(l+0x100,4);
  vid_puts("   00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F   0123456789ABCDEF");

  vid_putcurs(3,21); vid_puts("Use <F1> and <F2> to mark start and end address, then specify data type by");
  vid_putcurs(3,22); vid_puts("pressing <A>, <B>, <C> or <W> (");
  vid_attrib=mc[3]; vid_puts("ASCII"); vid_attrib=nc; vid_putch('/');
  vid_attrib=mc[2]; vid_puts("Byte"); vid_attrib=nc; vid_putch('/');
  vid_attrib=mc[0]; vid_puts("Code"); vid_attrib=nc; vid_putch('/');
  vid_attrib=mc[1]; vid_puts("Word"); vid_attrib=nc; vid_puts("). <ESC> for quit/save.");

  fread(buffer,1,256,cf);

  for(;;)
  {
    for(i=0, o=a+0x100; i<addrnum && o>=addrtab[i].addr; i++);

    vid_putcurs(0,4), vid_attrib=nc;

    for(y=0; y<16; y++)
    {
      vid_puts("   "), vid_puthex(a+(y<<4)+0x100,4), vid_puts("   ");

      for(x=0, i1=i, o=a+0x100+(y<<4); x<16; x++, o++)
      {
        if(i<addrnum && o>=addrtab[i].addr) i++; n=i?addrtab[i-1].type:C;
        vid_attrib=(x==xc && y==yc) ? cc : mc[n];
        if(a+(y<<4)+x<l) vid_puthex(buffer[y][x],2);
        else vid_puts("  ");
        vid_attrib=nc, vid_putch(' ');
      }
      vid_puts("  ");

      for(x=0, i=i1, o=a+0x100+(y<<4); x<16; x++, o++)
      {
        if(i<addrnum && o>=addrtab[i].addr) i++; n=i?addrtab[i-1].type:C;
        vid_attrib=(x==xc && y==yc) ? cc : mc[n];
        vid_putch((a+(y<<4)+x<l) ? buffer[y][x] : ' ');
        vid_attrib=nc;
      }
      vid_puts("    ");
    }

    o=a+0x100+(yc<<4)+xc;

    if(!addrnum) c_start=0x100, c_end=0x100+l-1, c_type=C;
    else
    {
      for(i=0; i<addrnum && o>=addrtab[i].addr; i++);
      if(!i) c_start=0x100, c_end=addrtab[i].addr-1, c_type=C;
      else
      {
        c_start=addrtab[i-1].addr, c_type=addrtab[i-1].type;
        if(i==addrnum) c_end=0x100+l-1;
        else c_end=addrtab[i].addr-1;
      }
    }

    vid_attrib=tc;
    vid_putcurs(17,24), vid_puthex(c_start,4);
    vid_putcurs(22,24), vid_puthex(c_end,4);
    vid_putcurs(27,24), vid_puts(types[c_type]);

    vid_putcurs(45,24);
    if(m_start) vid_puthex(m_start,4), vid_putch('-'), vid_puthex(m_end,4);

    switch(c=tolower(getch()))
    {
      case 0: switch(getch())
      {
        case 73: /* <PgUp> */
          if(a) fseek(cf,a-=256,SEEK_SET), fread(buffer,1,256,cf); break;

        case 81: /* <PgDn> */
          if(a+256<l) fseek(cf,a+=256,SEEK_SET), fread(buffer,1,256,cf); break;

        case 71: /* <Home> */
          fseek(cf,a=0,SEEK_SET), fread(buffer,1,256,cf); break;

        case 79: /* <End> */
          fseek(cf,a=(l-1)&0xFF00,SEEK_SET), fread(buffer,1,256,cf); break;

        case 72: /* <CuUp> */
          if(yc) yc--;
          else if(a) fseek(cf,a-=256,SEEK_SET), fread(buffer,1,256,cf), yc=15;
          break;

        case 80: /* <CuDn> */
          if(yc<15) yc++;
          else if(a+256<l) fseek(cf,a+=256,SEEK_SET), fread(buffer,1,256,cf), yc=0;
          break;

        case 75: /* <CuLe> */
          if(xc) xc--;
          else if(yc) yc--, xc=15;
          else if(a) fseek(cf,a-=256,SEEK_SET), fread(buffer,1,256,cf), yc=15, xc=15;
          break;

        case 77: /* <CuRi> */
          if(xc<15) xc++;
          else if(yc<15) yc++, xc=0;
          else if(a+256<l) fseek(cf,a+=256,SEEK_SET), fread(buffer,1,256,cf), yc=0, xc=0;
          break;

        case 59: /* <F1> */
          if(o<l+0x100) { m_start=o; if(m_end<o) m_end=o; } break;

        case 60: /* <F2> */
          if(o<l+0x100) { m_end=o; if(m_start>o) m_start=o; } break;
      } break;

      case 27: _setcursortype(_NORMALCURSOR); clrscr(); return;

      default:
        switch(c)
        {
          case 'c': m_type=C; break;
          case 'b': m_type=B; break;
          case 'a': m_type=A; break;
          case 'w': m_type=W; break;
        }
        if(c=='c' || c=='b' || c=='a' || c=='w')
          addaddr(m_start, m_end+1, m_type, l+0x100);
    }
  }
}

/*****************************************************************************
Hilfstext ausgeben und beenden:
*****************************************************************************/
void print_help(void)
{
  fprintf(stderr, "\n"

    "Generates A86 compatible 80x86/87 assembly code text from executable .COM files\n"
    "Invocation Syntax: R86 [-d] [-s] infile[.COM] [outfile[.ASM]]\n\n"

    "  infile    name of executable file to be disassembled\n"
    "  outfile   name of assembly code text file\n\n"

    "  -d        dump input file to determine the location of code and data areas\n"
    "            and save data in a file named infile.R86\n\n"

    "  -s        split output file into outfile.001, outfile.002, ... (each 60 KB)\n\n"

    "If invoked, R86 looks for an ASCII file named infile.R86 which contains a\n"
    "sorted list of hexadecimal addresses to specify code and data areas, e.g.\n\n"

    "  XXXX a    ASCII data begins at XXXXh\n"
    "  XXXX b    Byte data begins at XXXXh\n"
    "  XXXX c    Code (instructions) begins at XXXXh\n"
    "  XXXX w    Word data begins at XXXXh\n\n"

    "This file will be created by R86 if you use the -d switch.\n"
    "You may copy or distribute this program free of charge.\n");

  exit(1);
}

/*****************************************************************************
Int-23h-Handler:
*****************************************************************************/
void userbreak(void)
{
  fprintf(stderr,"Stalled.\n");
  fcloseall(), unlink(asmfile), exit(1);
}

/*****************************************************************************
Warten auf 'y' oder 'n'
*****************************************************************************/
int yn(void)
{
  BYTE c;
  do c=tolower(getch()); while(c!='y' && c!='n' && c!=13 && c!=27);
  return(c=='y' || c==13);
}

/*****************************************************************************
Kommandozeile analysieren, Dateien ffnen, ggf. Adredatei lesen
*****************************************************************************/
void initialize(int argc, char *argv[])
{
  FILE *fp;
  BYTE *c;
  int n, i, l, x;

  signal(SIGINT,userbreak);

  fprintf(stderr, "R86 Reassembler V1.00  Copyright (c) 1992 Stefan Bion\n");
  if(argc<2) print_help();

  for(n=1; n<argc; n++)
  {
    c=argv[n];
    if(*c=='-' || *c=='/')
    {
      switch(tolower(*(c+1)))
      {
        case 'd': d_switch=1; break;
        case 's': s_switch=1; break;
        default : print_help(); break;
      }
    }
    else break;
  }
  if(n==argc) print_help();

  l=strlen(strcpy(comfile,argv[n++]));
  for(i=l-1; i>l-5 && i>0 && !(x=(comfile[i]=='.')); i--);
  if(x) comfile[i]=0; strcpy(addrfile,comfile);

  if(x) comfile[i]='.'; else strcat(comfile,".COM");

  if(n==argc)
  {
    strcpy(asmfile,addrfile);
    if(!s_switch) strcat(asmfile,".ASM");
  }
  else
  {
    l=strlen(strcpy(asmfile,argv[n]));
    for(i=l-1; i>l-5 && i>0 && !(x=(asmfile[i]=='.')); i--);
    if(s_switch) { if(x) asmfile[i]=0; }
    else if(!x) strcat(asmfile,".ASM");
  }
  if(s_switch) ap=asmfile+strlen(asmfile), sprintf(ap,".%.3d", s_switch++);
  strupr(comfile), strupr(asmfile), strupr(strcat(addrfile,".R86"));

  if((fp=fopen(addrfile,"r"))!=NULL)
  {
    for(addrnum=0; addrnum<MAX_ADDR; addrnum++)
    {
      if(fscanf(fp,"%x %c",&addrtab[addrnum].addr,&addrtab[addrnum].type)!=EOF)
      {
        switch(addrtab[addrnum].type)
        {
          case 'c': addrtab[addrnum].type=C; break;
          case 'w': addrtab[addrnum].type=W; break;
          case 'b': addrtab[addrnum].type=B; break;
          case 'a': addrtab[addrnum].type=A; break;
          default :
            fprintf(stderr,"Error in file %s line %d\n", addrfile, addrnum+1);
            fcloseall(); exit(1);
        }
      }
      else break;
    }
    fclose(fp);
  }

  if((cf=fopen(comfile,"rb"))==NULL)
    fprintf(stderr,"Failed to open input file %s\n", comfile), exit(1);

  fseek(cf,0,SEEK_END), cflen=ftell(cf), fseek(cf,0,SEEK_SET);

  if(!cflen)
    fprintf(stderr,"Sorry, input file %s is empty\n", comfile),
    fcloseall(), exit(1);

  if(cflen>0xFF00)
    fprintf(stderr,"Sorry, input file %s is too big\n", comfile),
    fcloseall(), exit(1);

  if(d_switch)
  {
    dump();

    if(addrnum)
    {
      fprintf(stderr,"Save data into %s (y/n)? ", addrfile);
      n=yn(); fprintf(stderr,"\n");
      if(n)
      {
        if((fp=fopen(addrfile,"w"))!=NULL)
        {
          for(i=0; i<addrnum; i++)
          {
            switch(addrtab[i].type)
            {
              case C: x='c'; break;
              case W: x='w'; break;
              case B: x='b'; break;
              case A: x='a'; break;
            }
            fprintf(fp,"%x %c\n", addrtab[i].addr, x);
          }
          fclose(fp);
        }
        else fprintf(stderr,"Failed to open file %s\n", addrfile);
      }
    }

    fprintf(stderr,"Disassemble %s now (y/n)? ", comfile);
    n=yn(); fprintf(stderr,"\n"); if(!n) fcloseall(), exit(0);
  }

  if((af=fopen(asmfile,"w"))==NULL)
    fprintf(stderr,"Failed to open output file %s\n", asmfile),
    fcloseall(), exit(1);
}

/*****************************************************************************
Option -s: outfile.ASM in outfile.001 umbenennen oder neue Datei ffnen:
*****************************************************************************/
void split_file(void)
{
  if(pass && s_switch)
  {
    if(ftell(af)>0xF000)
    {
      fprintf(stderr,"- %lu bytes", ftell(af)); fclose(af);

      sprintf(ap,".%.3d", s_switch++);

      if((af=fopen(asmfile,"w"))==NULL)
        fprintf(stderr,"\nFailed to open output file %s\n",
        asmfile), fcloseall(), exit(1);

      fprintf(stderr, "\nWriting %s ", asmfile);
    }
  }
}

/*****************************************************************************
1 Byte aus der Datei lesen (gepuffert)
*****************************************************************************/
BYTE getbyte(void)
{
  static int f=0, n=0, i=0;
  static BYTE buffer[256];

  if(!f)
  {
    f=1;
    if(!(n=fread(buffer,1,256,cf)))
    {
      weiter=0;
      return(0);
    }
    else
      return(buffer[i++]);
  }
  else
  {
    if(unget_cnt) return(unget_byte[unget_cnt--]);

    if(i==n)
    {
      i=0;
      if(!(n=fread(buffer,1,256,cf)))
      {
        weiter=0;
        return(0);
      }
    }
    return(buffer[i++]);
  }
}

/*****************************************************************************
1 Word aus der Datei lesen
*****************************************************************************/
WORD getword(void)
{
  struct S { BYTE lo, hi; } s;
  union U { WORD w; struct S s; } u;

  u.s.lo=getbyte();
  u.s.hi=getbyte();
  return(u.w);
}

/*****************************************************************************
1 Byte in den Eingabestrom zurckstellen
*****************************************************************************/
void ungetbyte(BYTE b)
{
  unget_byte[++unget_cnt]=b;
}

/*****************************************************************************
Umwandeln eines Unsigned-Int-Wertes in einen String:
*****************************************************************************/
BYTE *int2str(WORD i)
{
  static BYTE s[8]="0";
  BYTE *p=s+1;

  if(i<0x0A)
    ultoa(i,p,10);
  else
  {
    strcat(strupr(ultoa(i,p,16)),"h");
    if(*p>'9') p--;
  }
  return(p);
}

/*****************************************************************************
Modus (Code, Bytes, Words, ASCII) feststellen:
*****************************************************************************/
void getmode(void)
{
  while(addrcnt<addrnum && ip>=addrtab[addrcnt].addr)
    mode=addrtab[addrcnt++].type;
}

/*****************************************************************************
Word in Symboltabelle einfgen
*****************************************************************************/
BYTE *addsym(WORD w)
{
  static BYTE s[10];
  BYTE *p=s+4;
  int i, j;

  if(pass)
  {
    if(w>=0x100)
    {
      s[0]='0', s[1]='0', s[2]='0', s[3]='0';
      *(p-=(6-strlen(strcat(strupr(ultoa(w,p,16)),"h"))))='l';
    }
    else p=int2str(w);
  }
  else
  {
    *p=0;
    if(w>=0x100)
    {
      if(!symnum) *symtab=w, symnum++;
      else
      {
        for(i=0; i<symnum && w>symtab[i]; i++);
        if(i<symnum && w==symtab[i]) return(p);
        if(symnum>=MAX_SYM)
          fprintf(stderr,"\nError: Too many symbols\n"),
          fcloseall(), unlink(asmfile), exit(1);
        if(i==symnum) {symtab[symnum++]=w; return(p);}
        for(j=symnum; j>i; j--) symtab[j]=symtab[j-1];
        symnum++; symtab[i]=w; return(p);
      }
    }
  }
  return(p);
}

/*****************************************************************************
Testen, ob aktueller IP >= nchstes Symbol in symtab:
*****************************************************************************/
int test_symbol(void)
{
  int i=0;

  if(pass) if(symcnt<symnum) i=(ip>=symtab[symcnt]);
  return(i);
}

/*****************************************************************************
Positionieren auf nchste Zeile/Spalte 20, ggf. Symbol(e) ausgeben:
*****************************************************************************/
void print_symbol(int f)
{
  WORD w;

  if(pass)
  {
    if(test_symbol())
    {
      if(multiple) multiple=0, fprintf(af,"\n");
      while(test_symbol())
      {
        w=symtab[symcnt], symcnt++;
        fprintf(af,"l%.4Xh%s            ", w, f&&ip==w?":":" ");
        if(ip>w) fprintf(af,"equ $-%s\n", int2str(ip-w));
      }
      if(ip>w) fprintf(af,"                   ");
    }
    else if(!multiple) fprintf(af,"                   ");
  }
}

/*****************************************************************************
Ausgeben der restlichen Symbole hinter dem Programmende:
*****************************************************************************/
void print_symbols(void)
{
  WORD w;

  if(pass)
  {
    while(symcnt<symnum)
    {
      w=symtab[symcnt++];

      fprintf(af,"l%.4Xh             equ $", w);
      if(w<=ip)
        fprintf(af,"-%s", int2str(ip-w));
      else
        fprintf(af,"+%s", int2str(w-ip));
      fprintf(af,"\n");
    }
  }
}

/*****************************************************************************
Falls Zeichen druckbar, dieses als Kommantar ausgeben
*****************************************************************************/
int isasc(BYTE c)
{
  return((c>=32 && c<127 && c!=39) || c==132 || c==148 ||
     c==129 || c==142 || c==153 || c==154 || c==225);
}

int isascg(BYTE c)      /* Zustzlich Grafikzeichen */
{
  return((c>=169 && c<=223) || isasc(c));
}

void print_asc(WORD b)
{
  BYTE c=b, d=b>>8;

  if(isasc(c))
  {
    if(d)
    {
      if(isasc(d)) fprintf(af,"   ;\'%c%c\'", d,c);
    }
    else
      fprintf(af,"   ;'%c'", c);
  }
}

/*****************************************************************************
Ausdrucken der Daten (ggf. mit DUP-Operator)
*****************************************************************************/
void print_dup(int m)
{
  BYTE s[20];
  int l;

  if(dupcnt)
  {
    if(dupcnt==1)
      l=strlen(strcpy(s,addsym(dupval)));
    else
      strcat(strcpy(s,int2str(dupcnt))," dup ("),
      l=strlen(strcat(strcat(s,addsym(dupval)),")"));

    if(data_len+l>57)
    {
      fprintf(af,"\n"); if(++linecnt>10) linecnt=0, split_file();
      fprintf(af,"                   d%c ", m==W?'w':'b'), data_len=0;
    }

    if(data_len) fprintf(af,","), data_len++;
    fprintf(af,"%s",s), data_len+=l, dupcnt=0;
  }
}

/*****************************************************************************
Ausdrucken von Datendefinitionen (Modi A, B und W)
*****************************************************************************/
void print_data(BYTE b)
{
  WORD w;
  BYTE *s;

  if(multiple) multiple=0, fprintf(af,"\n");
  if(prefix_flag) prefix_flag=0, fprintf(af,"\n");

  if(last_mode!=mode || (last_mode==A && data_len>53) || test_symbol())
  {
    if(last_mode!=C)
    {
      if(last_mode==A && ascii_mode) fprintf(af,"\'");
      print_dup(last_mode); fprintf(af,"\n");
    }
    else
      if(!emptyline) fprintf(af,"\n");

    if(++linecnt>10) linecnt=0, split_file();
    last_mode=mode, data_len=0, ascii_mode=isascg(b);
    print_symbol(0), fprintf(af,"d%c ", mode==W?'w':'b');
    if(mode==A && ascii_mode) fprintf(af,"\'"), data_len++;
  }

  switch(mode)
  {
    case A: ip++;
      if(isascg(b))
      {
        if(!ascii_mode)
        {
          fprintf(af,"\n"); if(++linecnt>10) linecnt=0, split_file();
          fprintf(af,"                   db \'"), data_len=1;
        }
        ascii_mode=1, data_len++, fprintf(af,"%c", b);
      }
      else
      {
        if(ascii_mode) fprintf(af,"\'"), data_len++;
        if(data_len) fprintf(af,","), data_len++;
        ascii_mode=0; data_len+=strlen(s=int2str(b)), fprintf(af,"%s", s);
      }
      break;

    case B: ip++;
      if(dupcnt)
      {
        if(b==dupval) dupcnt++;
        else print_dup(mode), dupval=b, dupcnt++;
      }
      else
        dupval=b, dupcnt++;
      break;

    case W: ip+=2;
      w=(getbyte()<<8)|b;
      if(dupcnt)
      {
        if(w<0x100 && w==dupval) dupcnt++;
        else print_dup(mode), dupval=w, dupcnt++;
      }
      else
        dupval=w, dupcnt++;
      break;
  }
}

/*****************************************************************************
Ausdrucken der Adressierungsart aufgrund des ModRM-Bytes
*****************************************************************************/
void print_modrm(int md, int rm)
{
  BYTE b;
  switch(md)
  {
    case 0:
      if(rm==6) ip+=2, fprintf(af,"[%s]", addsym(getword()));
      else
        fprintf(af,"[%s]", modrmtab[rm]);
      break;
    case 1:
      ip++; b=getbyte();
      if(rm==6 && !b)
        fprintf(af,"[bp]", modrmtab[rm]);
      else
        fprintf(af,"[%s%c%s]", modrmtab[rm],
          b<0x80 ? '+' : '-', int2str(b<0x80 ? b : 0x100-b));
      break;
    case 2:
      ip+=2;
      fprintf(af,"[%s+%s]", modrmtab[rm], addsym(getword()));
      break;
  }
}

/*****************************************************************************
Ausdrucken der Parameter aufgrund des ModRM-Bytes
*****************************************************************************/
int print_rmx(int md, int rm, int par1, int par2)
{
  WORD a=0;

  ip=ip+2+(par2>>3);    /* Opcode + ModRM + par2 */

  if(md==3)
    fprintf(af,"%s", regtab[par1>>4][rm]);
  else
  {
    fprintf(af,"%c", par1==32 ? 'd' : par1==16 ? 'w' : 'b');
    print_modrm(md,rm);
  }

  if(par2)
  {
    if(par2==8)
      fprintf(af,",%s", int2str(a=getbyte()));
    else
      a=getword(),
      fprintf(af,",%s%s", a<0x100?"":_OFFSET, addsym(a));
  }
  return(a);
}

/*****************************************************************************
Ausdrucken eines 80x87-Befehls:
*****************************************************************************/
void print_fp(BYTE b)
{
  BYTE b1=getbyte();
  BYTE *s;
  int md=b1>>6, rg=(b1>>3)&7, rm=b1&7;

  ip++;

  switch(b)
  {
    case 0xD8:
      fprintf(af,"%s", fmnemtab8[rg]);
      if(md<3)
        fprintf(af," d"), print_modrm(md,rm);
      else
        if(!(rm==1 && (rg==2 || rg==3))) fprintf(af," 0,%d", rm);
      break;

    case 0xD9:
      if(md<3)
      {
        if(rg==1) goto fperror;
        fprintf(af,"%s ", fmnemtab9[rg]);
        switch(rg)
        {
          case 0:
          case 2:
          case 3: fprintf(af,"d"); break;
          case 5:
          case 7: fprintf(af,"w"); break;
        }
        print_modrm(md,rm);
      }
      else
      {
        switch(rg)
        {
          case 0: fprintf(af,"%s %d", _FLD, rm); break;
          case 1: fprintf(af,"%s", _FXCH); if(rm!=1) fprintf(af," 0,%d", rm); break;
          default:
            switch(b1)
            {
              case 0xD0: s=_FNOP; break;
              case 0xE0: s=_FCHS; break;
              case 0xE1: s=_FABS; break;
              case 0xE4: s=_FTST; break;
              case 0xE5: s=_FXAM; break;
              case 0xE8: s=_FLD1; break;
              case 0xE9: s=_FLDL2T; break;
              case 0xEA: s=_FLDL2E; break;
              case 0xEB: s=_FLDPI; break;
              case 0xEC: s=_FLDLG2; break;
              case 0xED: s=_FLDLN2; break;
              case 0xEE: s=_FLDZ; break;
              case 0xF0: s=_F2XM1; break;
              case 0xF1: s=_FYL2X; break;
              case 0xF2: s=_FPTAN; break;
              case 0xF3: s=_FPATAN; break;
              case 0xF4: s=_FXTRACT; break;
              case 0xF5: s=_FPREM1; break;
              case 0xF6: s=_FDECSTP; break;
              case 0xF7: s=_FINCSTP; break;
              case 0xF8: s=_FPREM; break;
              case 0xF9: s=_FYL2XP1; break;
              case 0xFA: s=_FSQRT; break;
              case 0xFB: s=_FSINCOS; break;
              case 0xFC: s=_FRNDINT; break;
              case 0xFD: s=_FSCALE; break;
              case 0xFE: s=_FSIN; break;
              case 0xFF: s=_FCOS; break;
              default: goto fperror;
            }
            fprintf(af,"%s", s);
            break;
        }
      }
      break;

    case 0xDA:
      if(md<3)
        fprintf(af,"%s d", fmnemtabA[rg]), print_modrm(md,rm);
      else
      {
        if(rg==5 && rm==1) fprintf(af,"%s", _FUCOMPP);
        else goto fperror;
      }
      break;

    case 0xDB:
      if(md<3)
      {
        if(rg==1 || rg==4 || rg==6) goto fperror;
        fprintf(af,"%s ", fmnemtabB[rg]);
        if(rg==0 || rg==2 || rg==3) fprintf(af,"d");
        print_modrm(md,rm);
      }
      else
      {
        switch(b1)
        {
          case 0xE0: s=_FENI; break;
          case 0xE1: s=_FDISI; break;
          case 0xE2: s=_FNCLEX; break;
          case 0xE3: s=_FNINIT; break;
          case 0xE8: s=_FBANK_0; break;
          case 0xEA: s=_FBANK_2; break;
          case 0xEB: s=_FBANK_1; break;
          case 0xF1: s=_F4X4; break;
          default: goto fperror;
        }
        fprintf(af,"%s", s);
      }
      break;

    case 0xDC:
      if(md<3)
        fprintf(af,"%s q", fmnemtab8[rg]), print_modrm(md,rm);
      else
      {
        if(rg==2 || rg==3) goto fperror;
        fprintf(af,"%s %d,0", fmnemtabC[rg], rm);
      }
      break;

    case 0xDD:
      if(md<3)
      {
        if(rg==1 || rg==5) goto fperror;
        fprintf(af,"%s ", fmnemtabD[rg]);
        switch(rg)
        {
          case 0:
          case 2:
          case 3: fprintf(af,"q"); break;
          case 7: fprintf(af,"w"); break;
        }
        print_modrm(md,rm);
      }
      else
      {
        if(rg==1 || rg>5) goto fperror;
        fprintf(af,"%s", fmnemtabE[rg]);
        if(!(rm==1 && (rg==4 || rg==5))) fprintf(af," %d", rm);
      }
      break;

    case 0xDE:
      if(md<3)
        fprintf(af,"%s w", fmnemtabA[rg]), print_modrm(md,rm);
      else
      {
        if(rg==3 && rm==1) fprintf(af,"%s", _FCOMPP);
        else
        {
          if(rg==2) goto fperror;
          fprintf(af,"%s", fmnemtabC[rg]);
          if(rm!=1) fprintf(af,"p %d,0", rm);
        }
      }
      break;

    case 0xDF:
      if(md<3)
      {
        if(rg==1) goto fperror;
        fprintf(af,"%s ", fmnemtabF[rg]);
        switch(rg)
        {
          case 0:
          case 2:
          case 3: fprintf(af,"w"); break;
          case 5:
          case 7: fprintf(af,"q"); break;
        }
        print_modrm(md,rm);
      }
      else
      {
        if(rg==4 && rm==0) fprintf(af,"%s", _FSTSW_AX);
        else goto fperror;
      }
      break;
  }
  fpopcode=1;
  goto fpende;

fperror:
  ungetbyte(b1), ip--, fprintf(af,"%s %s", _DB, int2str(b));

fpende:
  return;
}

/*****************************************************************************
Hauptprogramm
*****************************************************************************/
void main(int argc, char *argv[])
{
  BYTE b, b1, b2;
  WORD w1, w2;
  int i, id;
  int md, rg, rm;
  BYTE *s;

  initialize(argc, argv);

  for(pass=0; pass<2; pass++)
  {
    ip=0x100, weiter=1, prefix_flag=0, multiple=0, unget_cnt=0,
    mode=C, last_mode=C, addrcnt=0, linecnt=0, emptyline=0;

    if(!pass) fprintf(stderr, "Scanning for references ");
    else      fprintf(stderr, "Writing %s ", asmfile);
    fseek(cf,0,SEEK_SET); fflush(af); doswrite(pass);

    if(fpopcode) fprintf(af,"                   .287\n\n");

    if(patch83)  fprintf(af,"patch83            macro\n"
                            "                   org $-#1\n"
                            "                   db 81h\n"
                            "                   org $+#1-1\n"
                            "                   db 0\n"
                            "                   #em\n\n");
    b=getbyte(), getmode();

    while(weiter)
    {
      if(mode==C)
      {
        if(last_mode!=C)
        {
          if(last_mode==A && ascii_mode) fprintf(af,"\'");
          print_dup(last_mode); fprintf(af,"\n\n"); last_mode=C;
        }

        id=opcode[b].id;

        if(multiple)
        {
          if(b<0x60)
            b1=(id==mult ? (b<0x20 ? (10+((b&7)==7))<<3 : b&0xF8) : 0);
          else
          {
            if(b==0xFE)
            {
              ungetbyte(b1=getbyte());
              b1=((b1&0xF0)==0xC0 ? 0x40+(b1&8) : 0);
            }
            else b1=0;
          }
          if(multiple!=b1) multiple=0, fprintf(af,"\n");
        }

        if(!prefix_flag) print_symbol(1);
        prefix_flag=(id==prefix);

        switch(id)
        {
          case unknown: ip++;
            fprintf(af,"%s %s", opcode[b].mnem, int2str(b)); break;

          case floatingp: ip++;
            print_fp(b); break;

          case mnem_aa: ip+=2;
            fprintf(af,"%s", opcode[b].mnem);
            if((b1=getbyte())!=0x0A) fprintf(af," %s", int2str(b1));
            break;

          case fwait_op: ip++;
            if((w1=getword())==0xE4DB)
              ip+=2, fprintf(af,"%s", _FSETPM);
            else
              ungetbyte(w1>>8), ungetbyte(w1),
              fprintf(af,"%s", opcode[b].mnem);
            break;

          case implicit: ip++;
            fprintf(af,"%s", opcode[b].mnem); break;

          case mult: ip++;
            if(multiple)
              fprintf(af,",%s", b<0x20 ? sregtab[b>>3] : regtab[1][b&7]);
            else
              multiple=(b<0x20 ? (10+((b&7)==7))<<3 : b&0xF8),
              fprintf(af,"%s %s", opcode[b].mnem,
                b<0x20 ? sregtab[b>>3] : regtab[1][b&7]);
            break;

          case prefix: ip++;          /* Prefix: Kein CR/LF */
            fprintf(af,"%s ", opcode[b].mnem); break;

          case offset: ip+=3;         /* Opcodes A0...A3 */
            fprintf(af,"%s ", opcode[b].mnem); s=addsym(getword());
            switch(b)
            {
              case 0xA0: fprintf(af,"al,b[%s]", s); break;
              case 0xA1: fprintf(af,"ax,w[%s]", s); break;
              case 0xA2: fprintf(af,"b[%s],al", s); break;
              case 0xA3: fprintf(af,"w[%s],ax", s); break;
            }
            break;

          case address: ip+=5;
            w1=getword(); w2=getword();
            fprintf(af,"%s %s", opcode[b].mnem, int2str(w2));
            fprintf(af,":%s", int2str(w1)); break; /* Bug in printf() ! */

          case disp8: ip+=2;
            fprintf(af,"%s %s", opcode[b].mnem,
              addsym((b1=getbyte())<0x80 ? ip+b1 : ip-0x100+b1)); break;

          case disp16: ip+=3;
            fprintf(af,"%s %s", opcode[b].mnem, addsym(ip+getword())); break;

          case data8i: ip+=2;
            b1=getbyte(); fprintf(af,"%s%s%s", opcode[b].mnem,
              b1<0x80 ? "" : "-" , int2str(b1<0x80 ? b1 : 0x100-b1));
            print_asc(b1); break;

          case data8: ip+=2;
            fprintf(af,"%s%s", opcode[b].mnem, int2str(b1=getbyte()));
            switch(b)
            {
              case 0xCD:                             /* INT */
              case 0xE4:                             /* IN AL, */
              case 0xE5: break;                      /* IN AX, */

              case 0xE6: fprintf(af,",al"); break;   /* OUT */
              case 0xE7: fprintf(af,",ax"); break;   /* OUT */

              default  : print_asc(b1); break;
            }
            break;

          case data16: ip+=3;
            w1=getword();
            fprintf(af,"%s%s%s", opcode[b].mnem, w1<0x100?"":_OFFSET, addsym(w1));
            if(b==0x68 && w1<0x80) fprintf(af,"   ; word operand"); /* PUSH data16 */
            print_asc(w1); break;

          case data16_1: ip+=3;
            fprintf(af,"%s %s", opcode[b].mnem, int2str(getword())); break;

          case data16_data8: ip+=4;
            fprintf(af,"%s %s", opcode[b].mnem, int2str(getword()));
            fprintf(af,",%s", int2str(getbyte())); break;

          case reg16: ip++;
            fprintf(af,"%s%s", opcode[b].mnem, regtab[1][b&7]); break;

          case reg8_data8: ip+=2;
            fprintf(af,"%s %s,%s", opcode[b].mnem, regtab[0][b&7],
              int2str(b1=getbyte()));
            print_asc(b1); break;

          case reg16_data16: ip+=3;
            w1=getword();
            fprintf(af,"%s %s,%s%s", opcode[b].mnem, regtab[1][b&7],
              w1<0x100?"":_OFFSET, addsym(w1));
            print_asc(w1); break;

          case opcode0F:
            b1=getbyte();
            if(b1==4 || b1==5 || (b1>=7 && b1<=0x0F) || b1==0x21 ||
               (b1>=0x23 && b1<=0x25) || b1==0x27 || b1==0x29 ||
               (b1>=0x2B && b1<=0x30) || b1==0x32 || (b1>=0x34 && b1<=0x38) ||
               b1==0x3A || (b1>=0x3C && b1<=0xFE)) goto operr;

            switch(b1)
            {
              case 0x06: ip+=2; fprintf(af,"%s", _CLTS); break;
              case 0x20: ip+=2; fprintf(af,"%s", _ADD4S); break;
              case 0x22: ip+=2; fprintf(af,"%s", _SUB4S); break;
              case 0x26: ip+=2; fprintf(af,"%s", _CMP4S); break;
              default: b2=getbyte(); md=b2>>6; rg=(b2>>3)&7; rm=b2&7;
              switch(b1)
              {
                case 0x00:
                  if(rg>=6) {ungetbyte(b2); goto operr;}
                  fprintf(af,"%s ", mnemtab0F00[rg]);
                  ip++; print_rmx(md,rm,16,0); break;

                case 0x01:
                  if(rg==5 || rg==7 || (rg<=4 && md==3))
                    {ungetbyte(b2); goto operr;}
                  fprintf(af,"%s ", mnemtab0F01[rg]);
                  ip++; print_rmx(md,rm,16,0); break;

                case 0x02:
                case 0x03:
                  fprintf(af,"%s %s,", b1==2?_LAR:_LSL, regtab[1][rg]);
                  ip++; print_rmx(md,rm,16,0); break;

                default:
                  if(b1<0x20)
                  {
                    fprintf(af,"%s ", mnemtab0F10[((b1&0x0F)>>1)&3]);
                    ip++; print_rmx(md,rm,((b1&1)+1)<<3,0);
                    if(b1<0x18) fprintf(af,",cl");
                    else ip++, fprintf(af,",%s", int2str(getbyte()));
                  }
                  else
                  {
                    switch(b1)
                    {
                      case 0xFF: ip+=3;
                        fprintf(af,"%s %s", _CALL80, int2str(b2)); break;
                      case 0x28: ip++;
                        fprintf(af,"%s ", _ROL4), print_rmx(md,rm,8,0); break;
                      case 0x2A: ip++;
                        fprintf(af,"%s ", _ROR4), print_rmx(md,rm,8,0); break;
                      default:
                        if(md!=3) {ungetbyte(b2); goto operr;} ip+=3;
                        fprintf(af,"%s %s", (b1==0x31 || b1==0x39)?
                          _STOBITS:_LODBITS, regtab[0][rm]);
                        if(b1==0x31 || b1==0x33)
                          fprintf(af,",%s", regtab[0][rg]);
                        else
                          ip++, fprintf(af,",%s", int2str(getbyte()));
                    }
                  }
                  break;
              }
              break;
            }
            break;

          default:      /* mod reg r/m */
            b1=getbyte(); md=b1>>6; rg=(b1>>3)&7; rm=b1&7;

            switch(id)
            {
              case modregrm:
                fprintf(af,"%s ", opcode[b].mnem);
                i=b&3; if((b>=0x84 && b<=0x87) || b==0x63) i&=1;
                if(i<2)
                  print_rmx(md,rm,(i+1)<<3,0),
                  fprintf(af,",%s", regtab[i][rg]);
                else
                  i&=1,
                  fprintf(af,"%s,", regtab[i][rg]),
                  print_rmx(md,rm,(i+1)<<3,0);
                if((b==0x87 && md==3 && rg==0) || (b>0x87 && b1==6))
                  fprintf(af,"   ; opcode %.2X", b);
                break;

              case mod000rm:
                if(rg)
                  goto operr;
                else
                {
                  fprintf(af,"%s ", opcode[b].mnem);
                  if(b==0x8F) print_rmx(md,rm,16,0);
                  else
                  {
                    if(b==0xC6) print_asc(print_rmx(md,rm,8,8));
                    else /*C7*/ print_asc(print_rmx(md,rm,16,16));
                    if(md==3) fprintf(af,"   ; opcode %.2X", b);
                  }
                }
                break;

              case mod0srrm:
                if(rg>3)
                  goto operr;
                else
                {
                  fprintf(af,"%s ", opcode[b].mnem);
                  if(b==0x8C)
                    print_rmx(md,rm,16,0), fprintf(af,",%s", sregtab[rg]);
                  else
                    fprintf(af,"%s,", sregtab[rg]), print_rmx(md,rm,16,0);
                }
                break;

              case modr16rm:
                if(md==3)
                  goto operr;
                else
                {
                  fprintf(af,"%s %s,%c", opcode[b].mnem, regtab[1][rg],
                    b==0x8D ? 'w' : 'd');
                  ip+=2; print_modrm(md,rm);
                }
                break;

              case modxxxrm:

                if(b>=0x80 && b<=0x83)
                  fprintf(af,"%s ", mnemtab80[rg]);

                if((b>=0xC0 && b<=0xC1) || (b>=0xD0 && b<=0xD3))
                {
                  if(rg==6) goto operr;
                  fprintf(af,"%s ", mnemtabD0[rg]);
                }

                if(b==0xF6 || b==0xF7)
                {
                  if(rg==1) goto operr;
                  fprintf(af,"%s ", mnemtabF6[rg]);
                }

                switch(b)
                {
                  case 0x80:
                  case 0x82:
                    print_asc(print_rmx(md,rm,8,8));
                    if(md==3 && rm==0) fprintf(af,"   ; opcode %.2X", b);
                    break;

                  case 0x81:
                    i=ip, print_asc(w1=print_rmx(md,rm,16,16));
                    if(md==3 && rm==0) fprintf(af,"   ; opcode %.2X", b);
                    if(w1<0x80)
                    {
                      if(pass)
                        fprintf(af,"\n                   patch83 %d", ip-i-1);
                      else patch83=1;
                    }
                    break;

                  case 0x83:
                    ip++, print_rmx(md,rm,16,0);
                    if((b1=getbyte())<0x80)
                      fprintf(af,",%s", int2str(b1));
                    else
                      fprintf(af,",0FF%.2Xh", b1);
                    print_asc(b1);
                    if(md==3 && rm==0) fprintf(af,"   ; opcode %.2X", b);
                    break;

                  case 0xC0: print_rmx(md,rm,8,8); break;
                  case 0xC1: print_rmx(md,rm,16,8); break;
                  case 0xD0: print_rmx(md,rm,8,0); fprintf(af,",1"); break;
                  case 0xD1: print_rmx(md,rm,16,0); fprintf(af,",1"); break;
                  case 0xD2: print_rmx(md,rm,8,0); fprintf(af,",cl"); break;
                  case 0xD3: print_rmx(md,rm,16,0); fprintf(af,",cl"); break;

                  case 0xF6:
                    if(!rg)
                    {
                      print_asc(print_rmx(md,rm,8,8));
                      if(md==3 && rm==0) fprintf(af,"   ; opcode %.2X", b);
                    }
                    else print_rmx(md,rm,8,0); break;

                  case 0xF7:
                    if(!rg)
                    {
                      print_asc(print_rmx(md,rm,16,16));
                      if(md==3 && rm==0) fprintf(af,"   ; opcode %.2X", b);
                    }
                    else print_rmx(md,rm,16,0); break;

                  case 0xFE:
                    if(rg>=2) goto operr;
                    if(multiple) ip=ip+2, fprintf(af,",%s", regtab[0][rm]);
                    else
                    {
                      if(md==3) multiple=0x40+(b1&8);
                      fprintf(af,"%s ", mnemtabFE[rg]); print_rmx(md,rm,8,0);
                    }
                    break;

                  case 0xFF:
                    if(rg==7||((rg==3||rg==5)&&md==3)) goto operr;
                    fprintf(af,"%s ", mnemtabFF[rg]);
                    print_rmx(md,rm,(rg==3||rg==5)?32:16,0);
                    if(md==3 && rg<2) fprintf(af,"   ; opcode %.2X", b);
                    break;
                }
                break;

              case imul:
                fprintf(af,"%s %s,", _IMUL, regtab[1][rg]);
                if(!(md==3 && rg==rm)) print_rmx(md,rm,16,0), fprintf(af,",");
                if(b==0x69)
                {
                  ip+=2, fprintf(af,"%s", addsym(w1=getword()));
                  if(w1<0x80) fprintf(af,"   ; word operand");
                }
                else
                  ip++, b1=getbyte(), fprintf(af,"%s%s",
                    b1<0x80 ? "" : "-" , int2str(b1<0x80 ? b1 : 0x100-b1));
                break;
            }
            break;
        }
        goto op_ok;

operr:  ungetbyte(b1), ip++, fprintf(af,"%s %s", _DB, int2str(b));

op_ok:  if(!prefix_flag && !multiple) fprintf(af,"\n");

        switch(b)      /* Nach JMP, RET oder IRET ein CR/LF ausgeben */
        {
          case 0xC2: /* RET, IRET */
          case 0xC3:
          case 0xCA:
          case 0xCB:
          case 0xCF:
          case 0xE9: /* JMP */
          case 0xEA:
          case 0xEB: fprintf(af,"\n"), emptyline=1; break;
          case 0xFF: if(rg==4 || rg==5) fprintf(af,"\n"), emptyline=1; break;
          default  : emptyline=0; break;
        }
        if(++linecnt>10 && !prefix_flag && !multiple) linecnt=0, split_file();
      }
      else print_data(b);      /* mode != C */

      b=getbyte(), getmode();
    }                              /* Kompletter Durchgang fr einen Pass */

    if(last_mode!=C)
    {
      if(last_mode==A && ascii_mode) fprintf(af,"\'");
      print_dup(last_mode); fprintf(af,"\n\n");
    }
    else if(!emptyline) fprintf(af,"\n");

    print_symbols();

    if(!pass) fprintf(stderr,"- %u symbols\n", symnum);
    else      fprintf(stderr,"- %lu bytes\n", ftell(af));

  } /* Pass 1 und 2 */
  fcloseall(), exit(0);
}

/*****************************************************************************
                             Ende von R86.C
*****************************************************************************/
